package main

import (
	"context"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/backoff"
	"google.golang.org/grpc/balancer"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	personalsearch "a.yandex-team.ru/travel/avia/personalization/api/personal_search/v2"
	hcbalancer "a.yandex-team.ru/travel/library/go/grpcutil/client/healthcheck_balancer"
	"a.yandex-team.ru/travel/library/go/grpcutil/client/ypresolver"
)

func main() {
	maxprocs.AdjustAuto()
	logger, _ := zap.NewDeployLogger(log.DebugLevel)

	logger.Info("Starting example")
	defer func() {
		logger.Info("Done with example")
	}()

	ctx, cancel := context.WithTimeout(context.Background(), 30*time.Second)
	defer cancel()

	gConn, err := grpc.DialContext(
		ctx,
		ypresolver.BuildServiceFQDN("avia-personalization-testing.api.grpc"),
		grpc.WithDefaultServiceConfig("{\"loadBalancingPolicy\":\"healthcheck_balancer\"}"),
		grpc.WithResolvers(ypresolver.NewYPResolverBuilder()),
		// disable health-check, as healthcheck_balancer performs one itself
		grpc.WithDisableHealthCheck(),
		// options below are unrelated to the balancer, modify them as needed
		grpc.WithInsecure(),
		grpc.WithBlock(),
		grpc.WithConnectParams(
			grpc.ConnectParams{
				Backoff: backoff.Config{
					BaseDelay:  time.Duration(1000 * time.Second),
					Multiplier: 2.0,
					MaxDelay:   time.Duration(7 * time.Second),
				},
			},
		),
	)
	if err != nil {
		logger.Fatal("did not connect", log.Error(err))
	}
	defer gConn.Close()

	personalizationClient := personalsearch.NewPersonalizationServiceV2Client(gConn)
	for i := 0; i < 20; i++ {
		ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
		defer cancel()
		response, err := personalizationClient.GetHotelsSuggest(
			ctx,
			&personalsearch.TGetHotelsSuggestRequestV2{YandexUid: "123", SearchesLimit: 3},
		)
		if err != nil {
			logger.Error("could not send hotels suggests request", log.Error(err))
		} else {
			logger.Info("got a response", log.Any("response", response))
		}
		time.Sleep(1 * time.Second)
	}
}

func init() {
	logger, _ := zap.NewDeployLogger(log.DebugLevel)
	balancer.Register(
		hcbalancer.NewBalancerBuilder(
			hcbalancer.WithLogger(logger),
			hcbalancer.WithHealthCheckServiceName("personalization-api"),
			hcbalancer.WithHealthCheckInterval(time.Duration(3000*time.Millisecond)),
			hcbalancer.WithBalancingMethod(hcbalancer.BalancingMethodChooseClosest),
		),
	)
}
