package healthcheckbalancer

import (
	"encoding/json"
	"fmt"

	"google.golang.org/grpc/serviceconfig"
)

const MaxAllowedConnectionsPerHost = 50

var ErrZeroConnectionsPerHost = fmt.Errorf("ConnectionsPerHost must be positive")

type ErrToManyConnectionsPerHost struct {
	connections uint
}

func NewErrToManyConnectionsPerHost(connections uint) *ErrToManyConnectionsPerHost {
	return &ErrToManyConnectionsPerHost{connections: connections}
}

func (e ErrToManyConnectionsPerHost) Error() string {
	return fmt.Sprintf("ConnectionsPerHost=%d must be less than MaxAllowedConnectionsPerHost=%d", e.connections, MaxAllowedConnectionsPerHost)
}

type ServiceConfig struct {
	LoadBalancingPolicy string                            `json:"LoadBalancingPolicy"`
	LoadBalancingConfig serviceconfig.LoadBalancingConfig `json:"LoadBalancingConfig"`
}

type loadBalancingConfig struct {
	serviceconfig.LoadBalancingConfig `json:"-"`
	LoadBalancerConfigs               []loadBalancerConfig
}

type loadBalancerConfig struct {
	serviceconfig.LoadBalancingConfig `json:"-"`
	Name                              string
	ConnectionsPerHost                uint
}

func (l loadBalancingConfig) MarshalJSON() ([]byte, error) {
	res := make([]map[string]json.RawMessage, 0)
	for _, c := range l.LoadBalancerConfigs {
		raw, _ := json.Marshal(c)
		res = append(res, map[string]json.RawMessage{c.Name: raw})
	}
	return json.Marshal(res)
}

func (l *loadBalancerConfig) Validate() error {
	if l.ConnectionsPerHost == 0 {
		return ErrZeroConnectionsPerHost
	}
	if l.ConnectionsPerHost > MaxAllowedConnectionsPerHost {
		return NewErrToManyConnectionsPerHost(l.ConnectionsPerHost)
	}
	return nil
}

func NewLoadBalancingConfig(lbConfigs ...loadBalancerConfig) serviceconfig.LoadBalancingConfig {
	return &loadBalancingConfig{LoadBalancerConfigs: lbConfigs}
}

func NewLoadBalancerConfig(name string, connectionsPerHost uint) loadBalancerConfig {
	return loadBalancerConfig{Name: name, ConnectionsPerHost: connectionsPerHost}
}

func (s *ServiceConfig) MarshalToJSON() string {
	result, _ := json.Marshal(&s)
	return string(result)
}
