package healthcheckbalancer

import (
	"sort"
	"sync"

	"google.golang.org/grpc/balancer"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
)

type lexicographicallyFirstInstancePicker struct {
	subConns        []namedSubConn
	hostListManager hostListManager

	mu     sync.Mutex
	logger log.Logger
}

func (p *lexicographicallyFirstInstancePicker) Pick(info balancer.PickInfo) (balancer.PickResult, error) {
	p.mu.Lock()
	defer p.mu.Unlock()

	healthySubConns := make([]namedSubConn, len(p.subConns))
	nextIndex := 0
	for i := 0; i < len(p.subConns); i++ {
		if p.hostListManager.isHealthy(p.subConns[i].host) {
			healthySubConns[nextIndex] = p.subConns[i]
			nextIndex++
		}
	}
	healthySubConns = healthySubConns[:nextIndex]
	if len(healthySubConns) == 0 {
		p.logger.Debug("no healthy host has been found")
		return balancer.PickResult{}, status.Error(codes.Unavailable, "no healthy host has been found")
	}

	sort.SliceStable(
		healthySubConns, func(i, j int) bool {
			return healthySubConns[i].host < healthySubConns[j].host
		},
	)
	picked := healthySubConns[0]

	p.logger.Debug("Picked host", log.String("host", picked.host))
	return balancer.PickResult{SubConn: picked.subConn}, nil
}
