package healthcheckbalancer

import (
	"math/rand"
	"sync"

	"google.golang.org/grpc/balancer"
	"google.golang.org/grpc/balancer/base"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/containers"
)

type healthCheckPickerBuilder struct {
	hostListManager hostListManager
	logger          log.Logger
	balancingMethod BalancingMethod
}

func (b *healthCheckPickerBuilder) Build(info base.PickerBuildInfo) balancer.Picker {
	if len(info.ReadySCs) == 0 {
		return base.NewErrPicker(balancer.ErrNoSubConnAvailable)
	}
	var subconns []namedSubConn
	var hosts []string
	var usedHosts = containers.SetOf[string]()
	for subconn, addr := range info.ReadySCs {
		host := addr.Address.Addr
		subconns = append(subconns, namedSubConn{subconn, host})
		if !usedHosts.Contains(host) {
			usedHosts.Add(host)
			hosts = append(hosts, host)
		}
	}

	b.hostListManager.updateHosts(hosts)
	return b.choosePicker(subconns)
}

func (b *healthCheckPickerBuilder) choosePicker(subconns []namedSubConn) balancer.Picker {
	switch b.balancingMethod {
	case BalancingMethodChooseClosest:
		return newClosestInstancePicker(subconns, b.hostListManager, b.logger)
	case BalancingMethodChooseLexicographicallyFirst:
		return &lexicographicallyFirstInstancePicker{
			subConns:        subconns,
			hostListManager: b.hostListManager,
			mu:              sync.Mutex{},
			logger:          b.logger,
		}
	case BalancingMethodRoundRobin:
		return b.buildRoundRobinPicker(subconns)
	default:
		return b.buildRoundRobinPicker(subconns)
	}
}

func (b *healthCheckPickerBuilder) buildRoundRobinPicker(subconns []namedSubConn) balancer.Picker {
	return &roundRobinPicker{
		subConns:        subconns,
		next:            rand.Intn(len(subconns)),
		hostListManager: b.hostListManager,
		mu:              sync.Mutex{},
		logger:          b.logger,
	}
}
