package staticresolver

import (
	"strings"
	"time"

	"google.golang.org/grpc/resolver"
)

const StaticScheme = "static"

type ServiceHasNoHostsError struct{}

func (e ServiceHasNoHostsError) Error() string {
	return "no hosts listed"
}

type staticResolverBuilder struct {
	connectionTimeout time.Duration
}

func NewStaticResolverBuilder() resolver.Builder {
	srb := &staticResolverBuilder{
		connectionTimeout: 30 * time.Second,
	}

	return srb
}

func (b *staticResolverBuilder) Build(
	target resolver.Target, cc resolver.ClientConn, opts resolver.BuildOptions) (resolver.Resolver, error) {
	r := &staticResolver{
		cc: cc,
	}

	err := r.start(target.Endpoint)
	if err != nil {
		return nil, err
	}
	return r, nil
}

func (b *staticResolverBuilder) Scheme() string { return StaticScheme }

type staticResolver struct {
	addresses []string
	cc        resolver.ClientConn
}

func (r *staticResolver) start(hostsList string) error {
	hosts := strings.Split(hostsList, ",")

	r.addresses = []string{}
	for _, host := range hosts {
		if len(host) > 0 {
			r.addresses = append(r.addresses, host)
		}
	}

	if len(r.addresses) == 0 {
		return ServiceHasNoHostsError{}
	}

	addrs := make([]resolver.Address, len(r.addresses))
	for i, s := range r.addresses {
		addrs[i] = resolver.Address{Addr: s}
	}

	return r.cc.UpdateState(resolver.State{Addresses: addrs})
}

func (*staticResolver) ResolveNow(o resolver.ResolveNowOptions) {}

func (*staticResolver) Close() {}
