package metrics

import (
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"

	"a.yandex-team.ru/library/go/core/metrics"
)

func WithFullMethodEndpoint() func(*metricsInterceptor) {
	return func(m *metricsInterceptor) {
		m.endpointKey = fullMethodEndpoint
	}
}

func fullMethodEndpoint(info *grpc.UnaryServerInfo) string {
	return info.FullMethod
}

// WithEndpointKey specifies custom function for determining endpoint from request.
//
// EndpointKey must return string from small fixed set.
func WithEndpointKey(keyGetter func(info *grpc.UnaryServerInfo) string) func(*metricsInterceptor) {
	return func(m *metricsInterceptor) {
		m.endpointKey = keyGetter
	}
}

func defaultDurationBuckets() []time.Duration {
	return []time.Duration{
		10 * time.Millisecond,
		50 * time.Millisecond,
		250 * time.Millisecond,
		1 * time.Second,
		5 * time.Second,
		25 * time.Second,
	}
}

// WithDurationBuckets specifies buckets to be used for request duration histogram.
func WithDurationBuckets(buckets metrics.DurationBuckets) func(*metricsInterceptor) {
	return func(m *metricsInterceptor) {
		m.durationBuckets = buckets
	}
}

// WithCodesCount creates sensors like code_3_count for grpc code InvalidArgument,
// please use WithCodesCountPretty
func WithCodesCount(grpcCodes ...codes.Code) Option {
	return func(interceptor *metricsInterceptor) {
		interceptor.codes = grpcCodes
		interceptor.addCodeTag = false
	}
}

// WithCodesCountPretty creates sensor code_count with tag code = (InvalidArgument, NotFound, Internal, etc.)
func WithCodesCountPretty(grpcCodes ...codes.Code) Option {
	return func(interceptor *metricsInterceptor) {
		interceptor.codes = grpcCodes
		interceptor.addCodeTag = true
	}
}
