package server

import (
	"context"
	"fmt"
	"strings"
	"time"

	grpcLog "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	"go.uber.org/zap/zapcore"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/logging"
)

type ctxShortMethodNameKey struct{}

var GRPCShortMethodContextKey = &ctxShortMethodNameKey{}

func NewGrpcLogFieldsInterceptor(headerNameToLogName map[string]string, withShortMethodKey bool) grpc.UnaryServerInterceptor {
	return func(ctx context.Context, req interface{}, info *grpc.UnaryServerInfo, handler grpc.UnaryHandler) (interface{}, error) {
		if withShortMethodKey {
			split := strings.Split(info.FullMethod, ".")
			shortMethod := split[len(split)-1]
			ctx = context.WithValue(ctx, GRPCShortMethodContextKey, shortMethod)
		}

		options := make([]log.Field, 0, len(headerNameToLogName)+1)
		options = append(options, log.String(logging.KeyGRPCMethod, info.FullMethod))

		if md, ok := metadata.FromIncomingContext(ctx); ok {
			for headerName, logName := range headerNameToLogName {
				if headerValue, exists := md[headerName]; exists && len(headerValue) != 0 {
					options = append(options, log.String(logName, headerValue[0]))
				}
			}
		}
		ctx = ctxlog.WithFields(ctx, options...)
		return handler(ctx, req)
	}
}

func NewMessageProducerFunc(logger *zap.Logger) grpcLog.MessageProducer {
	return func(ctx context.Context, msg string, level zapcore.Level, code codes.Code, err error, duration zapcore.Field) {
		if grpcShortMethod := ctx.Value(GRPCShortMethodContextKey); grpcShortMethod != nil {
			msg = fmt.Sprintf("finished unary call %v with code %s", grpcShortMethod, code.String())
		}
		fields := []log.Field{
			log.String(logging.KeyGRPCCode, code.String()),
			log.Duration(logging.KeyGRPCDuration, time.Duration(duration.Integer)),
		}
		if err != nil {
			fields = append(fields, log.Error(err))
		}
		switch level {
		case zapcore.DebugLevel:
			ctxlog.Debug(ctx, logger, msg, fields...)
		case zapcore.InfoLevel:
			ctxlog.Info(ctx, logger, msg, fields...)
		case zapcore.WarnLevel:
			ctxlog.Warn(ctx, logger, msg, fields...)
		case zapcore.ErrorLevel:
			fallthrough
		case zapcore.PanicLevel:
			fallthrough
		case zapcore.DPanicLevel:
			ctxlog.Error(ctx, logger, msg, fields...)
		case zapcore.FatalLevel:
			ctxlog.Fatal(ctx, logger, msg, fields...)
		}
	}
}
