package server

import (
	"context"
	"fmt"
	"net"
	"runtime/debug"
	"strings"

	middleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpc_auth "github.com/grpc-ecosystem/go-grpc-middleware/auth"
	grpcLog "github.com/grpc-ecosystem/go-grpc-middleware/logging/zap"
	recovery "github.com/grpc-ecosystem/go-grpc-middleware/recovery"
	grpc_opentracing "github.com/grpc-ecosystem/go-grpc-middleware/tracing/opentracing"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	ghealth "google.golang.org/grpc/health"
	"google.golang.org/grpc/health/grpc_health_v1"
	"google.golang.org/grpc/reflection"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	yatvm "a.yandex-team.ru/library/go/yandex/tvm"
	grpcmetrics "a.yandex-team.ru/travel/library/go/grpcutil/metrics"
	"a.yandex-team.ru/travel/library/go/tvm"
)

type NoEnvGrpcConfig struct {
	Addr                        string
	AllowReflection             bool
	HeadersNameToLogNameMapping map[string]string
}

func (cfg *NoEnvGrpcConfig) ToGrpcConfig() *GrpcConfig {
	return &GrpcConfig{
		Addr:                        cfg.Addr,
		AllowReflection:             cfg.AllowReflection,
		HeadersNameToLogNameMapping: cfg.HeadersNameToLogNameMapping,
	}
}

type GrpcConfig struct {
	Addr                        string            `config:"grpc-addr,required"`
	AllowReflection             bool              `config:"grpc-allow-reflection,optional"`
	HeadersNameToLogNameMapping map[string]string `yaml:"headers-name-to-log-name-mapping"`
}

var DefaultGrpcConfig = GrpcConfig{
	Addr:                        "[::]:9001",
	AllowReflection:             true,
	HeadersNameToLogNameMapping: nil,
}

type ServiceRegisterer func(server *grpc.Server)

type GrpcServer struct {
	server       *grpc.Server
	healthServer *ghealth.Server
	cfg          GrpcConfig
	logger       *zap.Logger
}

type grpcServerBuilder struct {
	interceptors []grpc.UnaryServerInterceptor
	logger       *zap.Logger
	registerers  []ServiceRegisterer
	cfg          GrpcConfig
	healthServer *ghealth.Server
}

func DefaultRecoveryInterceptor(logger *zap.Logger) grpc.UnaryServerInterceptor {
	recoveryOptions := []recovery.Option{
		recovery.WithRecoveryHandlerContext(
			func(ctx context.Context, p interface{}) (err error) {
				err = status.Errorf(codes.Internal, "Panic in grpc: %v", p)
				msg := "Recovery in grpc"
				if grpcShortMethod := ctx.Value(GRPCShortMethodContextKey); grpcShortMethod != nil {
					msg = fmt.Sprintf("Recovery in %v", grpcShortMethod)
				}
				ctxlog.ContextFields(ctx)
				if panicError, ok := p.(error); ok {
					ctxlog.Error(ctx, logger, msg,
						log.Error(fmt.Errorf("%w: %s", panicError, string(debug.Stack()))))
				} else {
					ctxlog.Error(ctx, logger, msg,
						log.Error(fmt.Errorf("%w: %s", err, string(debug.Stack()))))
				}
				return err
			},
		),
	}
	return recovery.UnaryServerInterceptor(recoveryOptions...)
}

func DefaultTracingInterceptor(skipTracingMethodNameParts ...string) grpc.UnaryServerInterceptor {
	filterFunc := func(ctx context.Context, fullMethodName string) bool {
		for _, skipVal := range skipTracingMethodNameParts {
			if strings.Contains(fullMethodName, skipVal) {
				return false
			}
		}
		return true
	}
	option := grpc_opentracing.WithFilterFunc(filterFunc)
	return grpc_opentracing.UnaryServerInterceptor(option)
}

func DefaultLogInterceptor(logger *zap.Logger) grpc.UnaryServerInterceptor {
	return grpcLog.UnaryServerInterceptor(
		logger.L,
		grpcLog.WithMessageProducer(NewMessageProducerFunc(logger)),
		grpcLog.WithDurationField(grpcLog.DurationToDurationField),
	)
}

// NewCleanGrpcServerBuilder creates clean GrpcServerBuilder without default interceptors
func NewCleanGrpcServerBuilder(
	cfg GrpcConfig,
	logger *zap.Logger,
) *grpcServerBuilder {
	return &grpcServerBuilder{
		cfg:          cfg,
		logger:       logger,
		registerers:  make([]ServiceRegisterer, 0),
		interceptors: []grpc.UnaryServerInterceptor{},
	}
}

func NewGrpcServerBuilder(
	cfg GrpcConfig,
	logger *zap.Logger,
	skipTracingMethodNameParts ...string,
) *grpcServerBuilder {
	return &grpcServerBuilder{
		cfg:         cfg,
		logger:      logger,
		registerers: make([]ServiceRegisterer, 0),
		interceptors: []grpc.UnaryServerInterceptor{
			DefaultTracingInterceptor(skipTracingMethodNameParts...),
			NewGrpcLogFieldsInterceptor(cfg.HeadersNameToLogNameMapping, false),
			DefaultRecoveryInterceptor(logger),
			DefaultLogInterceptor(logger),
		},
	}
}

func (sb *grpcServerBuilder) WithInterceptors(interceptors ...grpc.UnaryServerInterceptor) *grpcServerBuilder {
	sb.interceptors = append(sb.interceptors, interceptors...)
	return sb
}

func (sb *grpcServerBuilder) WithTVM(client yatvm.Client, allowedIDs []yatvm.ClientID) *grpcServerBuilder {
	return sb.WithInterceptors(
		tvm.CheckServiceTicketInterceptor(client, tvm.WithAllowedClients(allowedIDs), tvm.WithLogger(sb.logger)),
	)
}

func (sb *grpcServerBuilder) WithOptionalTVM(client yatvm.Client, allowedIDs []yatvm.ClientID) *grpcServerBuilder {
	if client == nil {
		return sb
	}
	return sb.WithInterceptors(
		tvm.CheckServiceTicketInterceptor(client, tvm.WithAllowedClients(allowedIDs), tvm.WithLogger(sb.logger)),
	)
}

func (sb *grpcServerBuilder) WithAuth(authFunc grpc_auth.AuthFunc) *grpcServerBuilder {
	return sb.WithInterceptors(
		grpc_auth.UnaryServerInterceptor(authFunc))
}

func (sb *grpcServerBuilder) WithMetrics(registry metrics.Registry, opts ...grpcmetrics.Option) *grpcServerBuilder {
	return sb.WithInterceptors(
		grpcmetrics.NewGrpcMetricsInterceptor(registry, opts...),
	)
}

func (sb *grpcServerBuilder) WithRegisterers(registerers ...ServiceRegisterer) *grpcServerBuilder {
	sb.registerers = append(sb.registerers, registerers...)
	return sb
}

func (sb *grpcServerBuilder) WithHealthServer(healthServer *ghealth.Server) *grpcServerBuilder {
	sb.healthServer = healthServer
	return sb
}

func (sb *grpcServerBuilder) Build() *GrpcServer {
	server := grpc.NewServer(
		middleware.WithUnaryServerChain(sb.interceptors...),
	)
	if sb.cfg.AllowReflection {
		reflection.Register(server)
	}
	if sb.healthServer != nil {
		sb.registerers = append(
			sb.registerers,
			func(server *grpc.Server) { grpc_health_v1.RegisterHealthServer(server, sb.healthServer) },
		)
	}
	for _, register := range sb.registerers {
		register(server)
	}
	return &GrpcServer{server: server, healthServer: sb.healthServer, cfg: sb.cfg, logger: sb.logger}
}

func (s *GrpcServer) Run(ctx context.Context) error {
	go func() {
		doneChannel := ctx.Done()
		if doneChannel != nil {
			<-doneChannel
			s.logger.Info("Shutting down listener", log.String("address", s.cfg.Addr))
			s.server.GracefulStop()
		}
	}()
	if s.healthServer != nil {
		defer s.healthServer.Shutdown()
	}
	listener, err := net.Listen("tcp", s.cfg.Addr)
	if err != nil {
		s.logger.Error("Failed to listen", log.Error(err), log.String("address", s.cfg.Addr))
		return err
	}
	return s.server.Serve(listener)
}

func (s *GrpcServer) GracefulStop() {
	s.server.GracefulStop()
}

func NewDefaultGrpcServerBuilder(
	cfg GrpcConfig,
	registerers []ServiceRegisterer,
	logger *zap.Logger,
	tvmClient yatvm.Client,
	tvmAllowedIds []yatvm.ClientID,
	registry metrics.Registry,
) *grpcServerBuilder {
	return NewGrpcServerBuilder(cfg, logger).
		WithMetrics(registry).
		WithRegisterers(registerers...).
		WithOptionalTVM(tvmClient, tvmAllowedIds)
}
