package servingstatus

import (
	"context"
	"time"

	"github.com/jonboulle/clockwork"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
)

type Service struct {
	logger                log.Logger
	conditions            []func() bool
	serviceName           string
	servingStatusSetter   func(string, healthpb.HealthCheckResponse_ServingStatus)
	updateInterval        time.Duration
	clock                 clockwork.Clock
	previousServingStatus healthpb.HealthCheckResponse_ServingStatus
}

func (s *Service) MonitorServingStatus(ctx context.Context) {
	go func() {
		for {
			if ctx.Err() == context.Canceled {
				s.logger.Infof("monitoring serving status for service %s has been cancelled", s.serviceName)
				return
			}
			servingStatus := s.getServingStatus()
			s.servingStatusSetter(s.serviceName, servingStatus)
			if servingStatus != s.previousServingStatus {
				if servingStatus == healthpb.HealthCheckResponse_SERVING {
					s.logger.Infof("service %s is serving", s.serviceName)
				} else if servingStatus == healthpb.HealthCheckResponse_NOT_SERVING {
					s.logger.Infof("service %s isn't serving", s.serviceName)
				} else if servingStatus == healthpb.HealthCheckResponse_UNKNOWN {
					s.logger.Infof("service %s is in UNKNOWN status", s.serviceName)
				}
			}
			s.previousServingStatus = servingStatus
			s.clock.Sleep(s.updateInterval)
		}
	}()
}

func (s *Service) getServingStatus() (result healthpb.HealthCheckResponse_ServingStatus) {
	defer func() {
		if r := recover(); r != nil {
			result = healthpb.HealthCheckResponse_UNKNOWN
		}
	}()
	for _, c := range s.conditions {
		if !c() {
			return healthpb.HealthCheckResponse_NOT_SERVING
		}
	}
	return healthpb.HealthCheckResponse_SERVING
}

func NewService(
	l log.Logger,
	serviceName string,
	servingStatusSetter func(string, healthpb.HealthCheckResponse_ServingStatus),
	updateInterval time.Duration,
	clock clockwork.Clock,
) *Service {
	return &Service{
		logger:                l,
		conditions:            make([]func() bool, 0),
		serviceName:           serviceName,
		servingStatusSetter:   servingStatusSetter,
		updateInterval:        updateInterval,
		clock:                 clock,
		previousServingStatus: healthpb.HealthCheckResponse_UNKNOWN,
	}
}

func (s *Service) Requires(condition func() bool) *Service {
	s.conditions = append(s.conditions, condition)
	return s
}
