package tracing

import (
	"context"
	"github.com/opentracing/opentracing-go"
	"google.golang.org/grpc"
)

type Option func(*options)

//FilterFunc should return false if we didn't want to trace request
type FilterFunc func(ctx context.Context, methodName string) bool

//RequestHandlerFunc for additional actions with request if they needed
type RequestHandlerFunc func(ctx context.Context, req interface{})

type options struct {
	filterFunc         FilterFunc
	requestHandlerFunc RequestHandlerFunc
}

func WithFilterFunc(f FilterFunc) Option {
	return func(o *options) {
		o.filterFunc = f
	}
}

func WithRequestHandlerFunc(f RequestHandlerFunc) Option {
	return func(o *options) {
		o.requestHandlerFunc = f
	}
}

func evaluateOptions(opts []Option) *options {
	newOptions := &options{}

	for _, o := range opts {
		o(newOptions)
	}

	return newOptions
}

func NewGrpcTracingInterceptor(options ...Option) grpc.UnaryServerInterceptor {
	o := evaluateOptions(options)
	return func(
		ctx context.Context,
		req interface{},
		info *grpc.UnaryServerInfo,
		handler grpc.UnaryHandler,
	) (interface{}, error) {
		fullMethod := info.FullMethod

		if o.filterFunc != nil && !o.filterFunc(ctx, fullMethod) {
			return handler(ctx, req)
		}

		span, ctx := opentracing.StartSpanFromContext(ctx, fullMethod)
		if o.requestHandlerFunc != nil {
			o.requestHandlerFunc(ctx, req)
		}
		defer span.Finish()

		resp, err := handler(ctx, req)

		return resp, err
	}
}
