package client

import (
	"container/list"
	"context"
	"sync"
	"time"

	"github.com/jonboulle/clockwork"
)

type ClientRequestStateRecord struct {
	state      int
	timestamp  time.Time
	requestKey interface{}
}

type ClientRequestStateStorage struct {
	mutex        sync.RWMutex
	defaultState int
	timeout      time.Duration
	queue        *list.List
	cache        map[interface{}]ClientRequestStateRecord
	clock        clockwork.Clock
}

func NewClientStateStorage(defaultState int, timeout time.Duration) *ClientRequestStateStorage {
	return NewClientStateStorageWithClock(defaultState, timeout, clockwork.NewRealClock())
}

func NewClientStateStorageWithClock(defaultState int, timeout time.Duration, clock clockwork.Clock) *ClientRequestStateStorage {
	return &ClientRequestStateStorage{
		defaultState: defaultState,
		timeout:      timeout,
		queue:        list.New(),
		cache:        make(map[interface{}]ClientRequestStateRecord),
		clock:        clock,
	}
}

func (s *ClientRequestStateStorage) Run(ctx context.Context) {
	go func() {
		ticker := s.clock.NewTicker(s.timeout)
		for {
			select {
			case <-ticker.Chan():
				s.mutex.Lock()
				threshold := s.clock.Now().Add(-s.timeout)
				item := s.queue.Front()
				for item != nil {
					record := item.Value.(ClientRequestStateRecord)
					if record.timestamp.After(threshold) {
						break
					}
					s.queue.Remove(item)
					recordFromCache, ok := (s.cache)[record.requestKey]
					// в кеше запись могла уже обновиться, проверяем, что удаляем соответствующие записи
					if ok && record.timestamp == recordFromCache.timestamp {
						delete(s.cache, record.requestKey)
					}
					item = item.Next() // эта операция валидна, даже если из списка был удален текущий элемент
				}
				s.mutex.Unlock()
			case <-ctx.Done():
				return
			}
		}
	}()
}

func (s *ClientRequestStateStorage) Set(requestKey interface{}, state int) {
	s.mutex.Lock()
	defer s.mutex.Unlock()
	record := ClientRequestStateRecord{state: state, timestamp: s.clock.Now(), requestKey: requestKey}
	s.queue.PushBack(record)
	(s.cache)[requestKey] = record
}

func (s *ClientRequestStateStorage) Get(requestKey interface{}) int {
	s.mutex.RLock()
	defer s.mutex.RUnlock()

	if record, ok := (s.cache)[requestKey]; ok {
		if record.timestamp.Add(s.timeout).After(s.clock.Now()) {
			return record.state
		}
	}
	return s.defaultState
}
