package client

import (
	"bytes"
	"context"
	"encoding/json"
	"encoding/xml"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"reflect"
	"strconv"
	"time"

	"github.com/go-chi/chi/v5/middleware"
	"github.com/opentracing/opentracing-go"
	//"github.com/eapache/go-resiliency/breaker" TODO: uncomment circuitBreaker code after https://st.yandex-team.ru/PEERDIR-7

	"a.yandex-team.ru/library/go/core/log"
)

var retryableHTTPCodes = map[int]struct{}{http.StatusBadGateway: {}, http.StatusAccepted: {}}
var circuitBreakerHTTPCodes = map[int]struct{}{http.StatusInternalServerError: {}, http.StatusRequestTimeout: {}}

type ContentType int

var (
	ContentTypeJSON ContentType = 1
	ContentTypeXML  ContentType = 2
)

type AuthProvider interface {
	Authentificate(*http.Request) error
}

type HTTPBaseAuthProvider struct {
	login    string
	password string
}

func NewHTTPBaseAuthProvider(login string, password string) (*HTTPBaseAuthProvider, error) {
	return &HTTPBaseAuthProvider{
		login:    login,
		password: password,
	}, nil
}

func (p HTTPBaseAuthProvider) Authentificate(r *http.Request) error {
	r.SetBasicAuth(p.login, p.password)
	return nil
}

type OAuthAuthProvider struct {
	OAuthToken string
}

func (p OAuthAuthProvider) Authentificate(r *http.Request) error {
	r.Header.Set("Authorization", "OAuth "+p.OAuthToken)
	return nil
}

type HTTPClient struct {
	logger       log.Logger
	baseURL      *url.URL
	timeout      time.Duration
	transport    http.RoundTripper
	contentType  ContentType
	authProvider AuthProvider
	//circuitBreaker *breaker.Breaker
}

func NewHTTPClient(baseURL string, timeout time.Duration,
	circuitBreakerThreshold int, circuitBreakerTimeout time.Duration, contentType ContentType, logger log.Logger,
	authProvider AuthProvider,
) (*HTTPClient, error) {
	return NewHTTPClientWithTransport(baseURL, timeout, nil,
		circuitBreakerThreshold, circuitBreakerTimeout, contentType, logger, authProvider)
}

func NewHTTPClientWithTransport(baseURL string, timeout time.Duration, transport http.RoundTripper,
	circuitBreakerThreshold int, circuitBreakerTimeout time.Duration, contentType ContentType, logger log.Logger,
	authProvider AuthProvider,
) (*HTTPClient, error) {
	const funcName = "NewHTTPClientWithTransport"
	base, err := url.Parse(baseURL)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", funcName, err)
	}
	//var circuitBreaker *breaker.Breaker = nil
	//if circuitBreakerThreshold > 0 {
	// circuitBreaker = breaker.New(circuitBreakerThreshold, circuitBreakerThreshold, circuitBreakerTimeout)
	//}
	if contentType != ContentTypeJSON && contentType != ContentTypeXML {
		return nil, fmt.Errorf("%s: unknown contentType=%d", funcName, contentType)
	}
	return &HTTPClient{
		logger:       logger,
		baseURL:      base,
		timeout:      timeout,
		transport:    transport,
		contentType:  contentType,
		authProvider: authProvider,
		//circuitBreaker: circuitBreaker,
	}, nil
}

func (c *HTTPClient) request(requestType string, ctx context.Context, path string, request interface{}, header http.Header,
	response interface{}) error {
	const funcName = "HTTPClient.request"

	if requestType != http.MethodGet && requestType != http.MethodPost {
		return fmt.Errorf("%s: unsupported requestType=%s", funcName, requestType)
	}

	ref, err := url.Parse(path)
	if err != nil {
		return fmt.Errorf("%s: can not parse url path '%s'", funcName, path)
	}

	var req *http.Request
	if requestType == http.MethodGet {
		values := url.Values{}
		requestVal := reflect.ValueOf(request).Elem()
		tp := requestVal.Type()
		for i := 0; i < requestVal.NumField(); i++ {
			f := requestVal.Field(i)
			if f.IsZero() {
				continue
			}
			name := tp.Field(i).Tag.Get("json")
			switch f.Interface().(type) {
			case bool:
				values.Set(name, strconv.FormatBool(f.Bool()))
			case int, int8, int16, int32, int64:
				values.Set(name, strconv.FormatInt(f.Int(), 10))
			case uint, uint8, uint16, uint32, uint64:
				values.Set(name, strconv.FormatUint(f.Uint(), 10))
			case float32:
				values.Set(name, strconv.FormatFloat(f.Float(), 'f', 4, 32))
			case float64:
				values.Set(name, strconv.FormatFloat(f.Float(), 'f', 4, 64))
			case []byte:
				values.Set(name, string(f.Bytes()))
			case string:
				values.Set(name, f.String())
			case time.Time:
				values.Set(name, f.Interface().(time.Time).Format(time.RFC3339))
			case []string:
				for i := 0; i < f.Len(); i++ {
					values.Add(name, f.Index(i).String())
				}
			default:
				return fmt.Errorf("%s: unsupported field type for field=%s", funcName, tp.Field(i).Name)
			}
		}
		if len(values) > 0 {
			p := fmt.Sprintf("%s?%s", path, values.Encode())
			ref, err = url.Parse(p)
			if err != nil {
				return fmt.Errorf("%s: can not parse url path '%s': %w", funcName, p, err)
			}
		}
		u := c.baseURL.ResolveReference(ref)
		req, err = http.NewRequestWithContext(ctx, http.MethodGet, u.String(), nil)
	} else {
		u := c.baseURL.ResolveReference(ref)
		content, er := json.Marshal(request)
		if er != nil {
			return fmt.Errorf("%s: %w", funcName, er)
		}
		req, err = http.NewRequestWithContext(ctx, http.MethodPost, u.String(), bytes.NewReader(content))
	}

	if err != nil {
		return fmt.Errorf("%s: can not build request: %w", funcName, err)
	}

	c.logger.Debugf("%s: %v", funcName, req.URL)

	if span := opentracing.SpanFromContext(ctx); span != nil {
		err := opentracing.GlobalTracer().Inject(
			span.Context(),
			opentracing.HTTPHeaders,
			opentracing.HTTPHeadersCarrier(req.Header))
		if err != nil {
			c.logger.Errorf("%s: tracing injection error: %s", funcName, err.Error())
		}
	}

	if reqID := middleware.GetReqID(ctx); reqID != "" {
		req.Header.Set("X-Request-Id", reqID)
	}
	for k, values := range header {
		for _, v := range values {
			req.Header.Add(k, v)
		}
	}

	client := http.Client{Timeout: c.timeout, Transport: c.transport}

	if c.authProvider != nil {
		err := c.authProvider.Authentificate(req)

		if err != nil {
			return fmt.Errorf("%s: can not apply authentication data to request: %w", funcName, err)
		}
	}

	resp, err := client.Do(req)

	if err != nil {
		return NewNotRetryableError(err.Error())
	}

	if resp.StatusCode != http.StatusOK {
		if isRetryableError(resp) {
			return NewRetryableError(NewBadResponseCodeError(resp.StatusCode))
		}
		return NewNotRetryableErrorWrap(NewBadResponseCodeError(resp.StatusCode))
	}

	defer func() { _ = resp.Body.Close() }()
	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return NewNotRetryableError(fmt.Sprintf("%s: %s", funcName, err.Error()))
	}

	if c.contentType == ContentTypeJSON {
		err = json.Unmarshal(data, response)
	} else if c.contentType == ContentTypeXML {
		err = xml.Unmarshal(data, response)
	} else {
		err = fmt.Errorf("%s: unsupported contentType=%d", funcName, c.contentType)
	}

	if err != nil {
		return NewNotRetryableError(fmt.Sprintf("%s: %s", funcName, err.Error()))
	}

	return nil
}

func (c *HTTPClient) Get(ctx context.Context, path string, request interface{}, response interface{}) error {
	const funcName = "HTTPClient.get"
	return c.GetWithHeader(ctx, path, request, http.Header{}, response)
}

func (c *HTTPClient) GetWithHeader(ctx context.Context, path string, request interface{}, header http.Header, response interface{}) error {
	const funcName = "HTTPClient.getWithHeader"
	return c.request(http.MethodGet, ctx, path, request, header, response)
}

func (c *HTTPClient) Post(ctx context.Context, path string, request interface{}, response interface{}) error {
	const funcName = "HTTPClient.post"
	return c.PostWithHeader(ctx, path, request, http.Header{}, response)
}

func (c *HTTPClient) PostWithHeader(ctx context.Context, path string, request interface{}, header http.Header, response interface{}) error {
	const funcName = "HTTPClient.postWithHeader"
	header.Add("Content-Type", "application/json")
	return c.request(http.MethodPost, ctx, path, request, header, response)
}

func isRetryableError(resp *http.Response) bool {
	_, ok := retryableHTTPCodes[resp.StatusCode]
	return ok
}

//func isCircuitBreakerError(resp *http.Response) bool {
// _, ok := circuitBreakerHTTPCodes[resp.StatusCode]
// return ok
//}

type TransportMock func(req *http.Request) (*http.Response, error)

func (m TransportMock) RoundTrip(req *http.Request) (*http.Response, error) {
	return m(req)
}

type SimpleTransportMock struct {
	statusCode int
	respBody   []byte
	callback   func()
	requests   []*http.Request
}

func (m *SimpleTransportMock) RoundTrip(req *http.Request) (*http.Response, error) {
	m.requests = append(m.requests, req)
	if m.callback != nil {
		m.callback()
	}
	return &http.Response{
		StatusCode: m.statusCode,
		Body:       ioutil.NopCloser(bytes.NewBuffer(m.respBody)),
		Header:     make(http.Header),
	}, nil
}
