package server

import (
	"context"
	"fmt"
	"net/http"
	"runtime/debug"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	tvmutil "a.yandex-team.ru/library/go/httputil/middleware/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/travel/library/go/tracing"
)

type HTTPConfig struct {
	Addr           string `config:"http-addr,required"`
	LogAllRequests bool   `config:"log-all-requests"`
}

type RouteBuilder func(router chi.Router)

var DefaultHTTPConfig = HTTPConfig{
	Addr: "[::]:9000",
}

type HTTPServerBuilder struct {
	cfg                      HTTPConfig
	logger                   log.Logger
	routeBuilders            []RouteBuilder
	unprotectedRouteBuilders []RouteBuilder

	tvmClient                    tvm.Client
	tvmAllowedIds                []tvm.ClientID
	httpMetricsRegistry          metrics.Registry
	httpMetricsMiddlewareOptions []httpmetrics.MiddlewareOption
	middlewares                  []func(next http.Handler) http.Handler
	recoverer                    func(next http.Handler) http.Handler
}

func NewHTTPServerBuilder(cfg HTTPConfig, logger log.Logger, routeBuilders []RouteBuilder, httpMetricsRegistry metrics.Registry) *HTTPServerBuilder {
	return &HTTPServerBuilder{
		cfg:                      cfg,
		logger:                   logger,
		routeBuilders:            routeBuilders,
		unprotectedRouteBuilders: []RouteBuilder{},
		httpMetricsRegistry:      httpMetricsRegistry,
		httpMetricsMiddlewareOptions: []httpmetrics.MiddlewareOption{
			httpmetrics.WithPathEndpoint(),
			httpmetrics.WithSolomonRated(),
		},
	}
}

// These routes aren't going to be protected with TVM
func (b *HTTPServerBuilder) WithUnprotectedRoutes(unprotectedRouteBuilders []RouteBuilder) *HTTPServerBuilder {
	b.unprotectedRouteBuilders = append(b.unprotectedRouteBuilders, unprotectedRouteBuilders...)
	return b
}

func (b *HTTPServerBuilder) WithMiddlewares(middlewares ...func(next http.Handler) http.Handler) *HTTPServerBuilder {
	b.middlewares = append(b.middlewares, middlewares...)
	return b
}

func (b *HTTPServerBuilder) WithLoggingRecoverer() *HTTPServerBuilder {
	b.recoverer = func(next http.Handler) http.Handler {
		fn := func(w http.ResponseWriter, r *http.Request) {
			defer func() {
				if rvr := recover(); rvr != nil {
					if rvr == http.ErrAbortHandler {
						// we don't recover http.ErrAbortHandler so the response
						// to the client is aborted, this should not be logged
						panic(rvr)
					}
					b.logger.Errorf("Panic: %v\n %s\n", rvr, string(debug.Stack()))
					w.WriteHeader(http.StatusInternalServerError)
				}
			}()

			next.ServeHTTP(w, r)
		}
		return http.HandlerFunc(fn)
	}
	return b
}

func (b *HTTPServerBuilder) WithTVM(tvmClient tvm.Client, tvmAllowedIds []tvm.ClientID) *HTTPServerBuilder {
	b.tvmClient = tvmClient
	b.tvmAllowedIds = tvmAllowedIds
	return b
}

func (b *HTTPServerBuilder) WithMetricsMiddlewareOptions(options ...httpmetrics.MiddlewareOption) *HTTPServerBuilder {
	b.httpMetricsMiddlewareOptions = append(b.httpMetricsMiddlewareOptions, options...)
	return b
}

type wrappedLogger struct {
	logger log.Logger
}

func (wl wrappedLogger) Print(v ...interface{}) {
	// old version of chi.middleware only prints one element anyway,
	// and for the next one we shall switch to httplog.RequestLogger
	for _, elem := range v {
		wl.logger.Info(fmt.Sprint(elem))
	}
}

func (b *HTTPServerBuilder) Build() *HTTPServer {
	tm := tracing.NewTracingMiddlewareBuilder().
		WithExtractor(tracing.NewHeaderTagExtractor("request-id", "X-Request-ID")).
		Build()

	r := chi.NewRouter()
	if b.cfg.LogAllRequests {
		requestLogger := middleware.RequestLogger(
			&middleware.DefaultLogFormatter{
				Logger:  wrappedLogger{b.logger},
				NoColor: true,
			},
		)
		r.Use(requestLogger)
	}
	r.Use(middleware.RequestID)

	if b.recoverer == nil {
		b.recoverer = middleware.Recoverer
	}
	r.Use(b.recoverer)

	r.Use(b.middlewares...)

	for _, unprotectedRoute := range b.unprotectedRouteBuilders {
		unprotectedRoute(r)
	}

	rh := r.With(tm.Handle)
	if b.tvmClient != nil {
		rh.Use(tvmutil.CheckServiceTicket(b.tvmClient, tvmutil.WithAllowedClients(b.tvmAllowedIds), tvmutil.WithLogger(b.logger.Structured())))
	}
	rh.Use(httpmetrics.New(b.httpMetricsRegistry, b.httpMetricsMiddlewareOptions...))

	for _, buildRoute := range b.routeBuilders {
		buildRoute(rh)
	}

	return &HTTPServer{
		cfg:    b.cfg,
		router: r,
		logger: b.logger,
	}
}

type HTTPServer struct {
	cfg    HTTPConfig
	router chi.Router
	logger log.Logger
}

func (s *HTTPServer) Run(ctx context.Context) error {
	server := &http.Server{Addr: s.cfg.Addr, Handler: s.router}
	go func() {
		doneChannel := ctx.Done()
		if doneChannel != nil {
			<-doneChannel
			s.logger.Info("Shutting down listener", log.String("address", s.cfg.Addr))
			_ = server.Shutdown(context.Background())
		}
	}()
	err := server.ListenAndServe()
	if ctx.Err() == context.Canceled && err == http.ErrServerClosed {
		return nil // serverClosed is not an error if context is cancelled, we are just exiting
	} else {
		return err
	}
}

func RunHTTPServer(ctx context.Context, cfg HTTPConfig, routeBuilders []RouteBuilder, logger log.Logger, tvmClient tvm.Client, tvmAllowedIds []tvm.ClientID, httpMetricsRegistry metrics.Registry) error {
	server := NewHTTPServerBuilder(cfg, logger, routeBuilders, httpMetricsRegistry).
		WithTVM(tvmClient, tvmAllowedIds).
		Build()

	return server.Run(ctx)
}
