package logbroker

import (
	"fmt"

	"a.yandex-team.ru/kikimr/public/sdk/go/persqueue/recipe"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/ydb/auth/tvm"
	"a.yandex-team.ru/travel/library/go/vault"
)

const (
	newYavCredentialsProviderFn         = "library.go.logbroker.NewYavCredentialsProvider"
	newTVMCredentialsProviderFn         = "library.go.logbroker.NewTVMCredentialsProvider"
	yavCredentialsProviderCredentialsFn = "library.go.logbroker.YavCredentialsProvider.Credentials"
	tvmCredentialsProviderCredentialsFn = "library.go.logbroker.TVMCredentialsProvider.Credentials"
)

type OAuthCredentialsProvider struct {
	token string
}

func NewOAuthCredentialsProvider(token string) *OAuthCredentialsProvider {
	return &OAuthCredentialsProvider{token: token}
}

func (p *OAuthCredentialsProvider) Credentials() (ydb.Credentials, error) {
	return ydb.AuthTokenCredentials{AuthToken: p.token}, nil
}

type YavCredentialsProvider struct {
	tokenKey       string
	tokenSecret    string
	secretResolver *vault.YavSecretsResolver
}

func NewYavCredentialsProvider(tokenKey string, tokenSecret string, secretResolver *vault.YavSecretsResolver) *YavCredentialsProvider {
	if secretResolver == nil {
		panic(fmt.Sprintf("%s: secretResolver == nil", newYavCredentialsProviderFn))
	}

	return &YavCredentialsProvider{
		tokenKey:       tokenKey,
		tokenSecret:    tokenSecret,
		secretResolver: secretResolver,
	}
}

func (p *YavCredentialsProvider) Credentials() (ydb.Credentials, error) {
	oauthToken, err := p.secretResolver.GetSecretValue(p.tokenSecret, p.tokenKey)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", yavCredentialsProviderCredentialsFn, err)
	}
	return ydb.AuthTokenCredentials{AuthToken: oauthToken}, nil
}

type TVMCredentialsProvider struct {
	selfID    uint32
	serviceID uint32
	secret    string
	logger    log.Logger
}

func NewTVMCredentialsProvider(selfID, serviceID uint32, secret string, logger log.Logger) *TVMCredentialsProvider {
	if logger == nil {
		panic(fmt.Sprintf("%s: logger == nil", newTVMCredentialsProviderFn))
	}

	return &TVMCredentialsProvider{
		selfID:    selfID,
		serviceID: serviceID,
		secret:    secret,
		logger:    logger,
	}
}

func (p *TVMCredentialsProvider) Credentials() (ydb.Credentials, error) {
	credentials, err := tvm.Client(tvm.ClientConfig{
		SelfID:    p.selfID,
		ServiceID: p.serviceID,
		Secret:    p.secret,
	}, p.logger)
	if err != nil {
		return nil, fmt.Errorf("%s: %w", tvmCredentialsProviderCredentialsFn, err)
	}
	return credentials, nil
}

type TestCredentialsProvider struct {
	env *recipe.Env
}

func NewTestCredentialsProvider(env *recipe.Env) *TestCredentialsProvider {
	return &TestCredentialsProvider{
		env: env,
	}
}

func (p *TestCredentialsProvider) Credentials() (ydb.Credentials, error) {
	return p.env.ConsumerOptions().Credentials, nil
}
