package multi

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/cenkalti/backoff/v4"
	"go.uber.org/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/metrics"
	"a.yandex-team.ru/travel/library/go/syncutil"
)

type ConsumerConfig struct {
	Topic                 TopicConfig `yaml:"topic"`
	ReadEndpoints         []string    `yaml:"read_endpoints"`
	MinActiveConsumers    int32       `yaml:"min_active_consumers"`
	Name                  string      `yaml:"name"`
	DecompressionDisabled bool        `yaml:"decompression_disabled"`
	Multiline             bool        `yaml:"multiline"`
}

type MultiEndpointConsumer struct {
	logger          log.Logger
	config          ConsumerConfig
	name            string
	activeConsumers atomic.Int32
	token           string
	running         bool
}

func NewMultiEndpointConsumer(
	logger log.Logger,
	config ConsumerConfig,
	token string,
) *MultiEndpointConsumer {
	name := config.Name + "Consumer"
	return &MultiEndpointConsumer{
		logger: logger.WithName(name),
		config: config,
		name:   name,
		token:  token,
	}
}

func (c *MultiEndpointConsumer) Name() string {
	return c.name
}

func (c *MultiEndpointConsumer) Ready() bool {
	return c.config.MinActiveConsumers <= c.activeConsumers.Load()
}

func (c *MultiEndpointConsumer) Running() bool {
	return c.running
}

func (c *MultiEndpointConsumer) Run(ctx context.Context, dispatcher MessageDispatcher) {
	c.running = true
	defer func() { c.running = false }()
	wg := syncutil.WaitGroup{}
	wg.Add(len(c.config.ReadEndpoints))
	for _, endpoint := range c.config.ReadEndpoints {
		targetEndpoint := endpoint
		wg.Go(func() {
			c.runSingleEndpoint(ctx, targetEndpoint, dispatcher)
		})
	}
	wg.Wait()
}

func (c *MultiEndpointConsumer) runSingleEndpoint(ctx context.Context, endpoint string, dispatcher MessageDispatcher) {
	_ = backoff.RetryNotify(
		func() error {
			select {
			case <-ctx.Done():
				return backoff.Permanent(fmt.Errorf("consumer has been stopped"))
			default:
			}
			consumerName := strings.Join([]string{c.name, endpoint}, "-")
			consumer, err := NewSingleEndpointConsumer(
				ctx,
				c.config,
				c.token,
				endpoint,
				c.logger,
				consumerName,
				dispatcher,
			)
			if err != nil {
				return fmt.Errorf("failed to start consumer %s: %w", consumerName, err)
			}
			metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, map[string]string{"endpoint": endpoint}, aliveConsumers).Set(1)
			defer metrics.GlobalAppMetrics().GetOrCreateGauge(metricsPrefix, map[string]string{"endpoint": endpoint}, aliveConsumers).Set(0)
			c.activeConsumers.Inc()
			err = consumer.Read()
			c.activeConsumers.Dec()
			if err != nil {
				c.logger.Error("consumer has been closed unexpectedly", log.Error(err))
				return err
			}
			return nil
		},
		backoff.NewConstantBackOff(10*time.Second),
		func(err error, duration time.Duration) {
			select {
			case <-ctx.Done():
				return
			default:
				c.logger.Error("failed to run single endpoint consumer. The operation will be retried", log.Error(err))
			}
		},
	)
}
