package logbroker

import (
	"context"
	"fmt"

	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/ytlock"
)

const (
	ytLockGroupAcquireFn = "library.go.logbroker.YtLockGroup.Acquire"
	ytLockGroupReleaseFn = "library.go.logbroker.YtLockGroup.Release"
)

type ytLockGroupOption func(lock *YtLockGroup)

func WithOnLostCallbacks(callbacks ...func()) ytLockGroupOption {
	return func(pool *YtLockGroup) {
		pool.onLostCallbacks = append(pool.onLostCallbacks, callbacks...)
	}
}

type YtLockGroup struct {
	ytClient yt.Client
	ytPrefix string
	poolSize uint

	onAcquireCallbacks []func()
	onReleaseCallbacks []func()
	onLostCallbacks    []func()

	lockedNumber uint
	lock         *ytlock.Lock
	stopWatch    chan struct{}
}

func NewYtLockGroup(ytClient yt.Client, ytPrefix string, poolSize uint, options ...ytLockGroupOption) *YtLockGroup {
	group := &YtLockGroup{
		ytClient:  ytClient,
		ytPrefix:  ytPrefix,
		poolSize:  poolSize,
		stopWatch: make(chan struct{}),
	}

	for _, option := range options {
		option(group)
	}

	return group
}

func (g *YtLockGroup) LockedNumber() uint {
	return g.lockedNumber
}

func (g *YtLockGroup) Acquire(ctx context.Context) error {
	if g.lock != nil {
		return fmt.Errorf("%s: group is already locked with path=%s", ytLockGroupAcquireFn, g.makeYtPath(g.lockedNumber))
	}

	var err error
	for i := uint(1); i < g.poolSize+1; i++ {
		lockPath := ypath.Path(g.makeYtPath(i))
		lock := ytlock.NewLock(g.ytClient, lockPath)

		var lost <-chan struct{}
		lost, err = lock.Acquire(ctx)
		if err == nil {
			g.lockedNumber = i
			g.lock = lock
			for _, callback := range g.onAcquireCallbacks {
				callback()
			}
			go g.watch(lost)
			return nil
		}
	}
	if err != nil {
		return fmt.Errorf("%s: %w", ytLockGroupAcquireFn, err)
	}

	go func() {
		<-ctx.Done()
		g.stopWatch <- struct{}{}
		g.lock = nil
	}()
	return nil
}

func (g *YtLockGroup) makeYtPath(number uint) string {
	return fmt.Sprintf("%s-%d", g.ytPrefix, number)
}

func (g *YtLockGroup) watch(lost <-chan struct{}) {
	select {
	case <-lost:
		g.lockedNumber = 0
		g.lock = nil
		for _, callback := range g.onLostCallbacks {
			callback()
		}
		return
	case <-g.stopWatch:
		return
	}
}
