package logging

import (
	"fmt"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/library/go/errorbooster"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

type Format int

const (
	QloudFormat Format = iota
	DeployFormat
)

type loggerBuilder struct {
	config             *Config
	withSentry         bool
	errorBoosterClient errorbooster.Client
	format             Format
}

type Option func(*loggerBuilder)

func NewLogger(opts ...Option) (*arczap.Logger, error) {
	builder := &loggerBuilder{
		config: &DefaultConfig,
		format: DeployFormat,
	}
	for _, opt := range opts {
		opt(builder)
	}
	level, err := log.ParseLevel(builder.config.Level)
	if err != nil {
		return nil, fmt.Errorf("bad log level format: %w", err)
	}
	var mainLogger *arczap.Logger
	switch builder.format {
	case QloudFormat:
		mainLogger, err = arczap.NewQloudLogger(level)
	case DeployFormat:
		mainLogger, err = arczap.NewDeployLogger(level)
	}
	if err != nil {
		return nil, fmt.Errorf("cannot create logger: %w", err)
	}
	cores := []zapcore.Core{mainLogger.L.Core()}
	if builder.withSentry {
		err = ConfigureSentry(builder.config.SentryDSN)
		if err != nil {
			return nil, fmt.Errorf("cannot configure logging to sentry: %w", err)
		}
		sentryLevel, err := log.ParseLevel(builder.config.SentryLevel)
		if err != nil {
			sentryLevel = log.ErrorLevel
		}
		cores = append(cores, NewSentryLoggerZapCore(arczap.ZapifyLevel(sentryLevel)))
	}
	if builder.errorBoosterClient != nil {
		ebLevel, err := log.ParseLevel(builder.config.ErrorBoosterLevel)
		if err != nil {
			ebLevel = log.ErrorLevel
		}
		cores = append(cores, NewErrorBoosterLoggerZapCore(
			arczap.ZapifyLevel(ebLevel),
			builder.errorBoosterClient,
			builder.config.ErrorBoosterAdditionalFields,
		))
	}
	if len(cores) > 1 {
		tee := zapcore.NewTee(cores...)
		return arczap.NewWithCore(tee, zap.WithCaller(true)), nil
	} else {
		return mainLogger, nil
	}
}

func WithSentry(withSentry bool) Option {
	return func(builder *loggerBuilder) {
		builder.withSentry = withSentry
	}
}

func WithConfig(config *Config) Option {
	return func(builder *loggerBuilder) {
		builder.config = config
	}
}

func WithFormat(format Format) Option {
	return func(builder *loggerBuilder) {
		builder.format = format
	}
}

func WithErrorBooster(client errorbooster.Client) Option {
	return func(builder *loggerBuilder) {
		builder.errorBoosterClient = client
	}
}
