package logging

import (
	"fmt"

	"a.yandex-team.ru/travel/library/go/errorbooster"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

func NewErrorBoosterLoggerZapCore(level zapcore.Level, client errorbooster.Client, additionalFieldsKeys []string) zapcore.Core {
	additionalFields := make(map[string]bool)
	for _, key := range additionalFieldsKeys {
		additionalFields[key] = true
	}
	return &errorBoosterLoggerZapCore{
		LevelEnabler:     zap.NewAtomicLevelAt(level),
		client:           client,
		additionalFields: additionalFields,
	}
}

type errorBoosterLoggerZapCore struct {
	zapcore.LevelEnabler
	fields           []zapcore.Field
	client           errorbooster.Client
	additionalFields map[string]bool
}

func (e errorBoosterLoggerZapCore) With(fields []zapcore.Field) zapcore.Core {
	return &errorBoosterLoggerZapCore{
		LevelEnabler:     e.LevelEnabler,
		fields:           append(e.fields, fields...),
		client:           e.client,
		additionalFields: e.additionalFields,
	}
}

func (e errorBoosterLoggerZapCore) Check(entry zapcore.Entry, checked *zapcore.CheckedEntry) *zapcore.CheckedEntry {
	if e.Enabled(entry.Level) {
		checked = checked.AddCore(entry, e)
	}
	return checked
}

func (e errorBoosterLoggerZapCore) Write(entry zapcore.Entry, fields []zapcore.Field) error {
	fields = append(fields, e.fields...)
	msg := errorbooster.Message{
		Message:    entry.Message,
		Stack:      entry.Stack,
		Timestamp:  entry.Time.UnixMilli(),
		Level:      entry.Level.String(),
		Additional: map[string]interface{}{},
	}
	if msg.Stack == "" {
		if err := findErrorInZapFields(fields); err != nil {
			msg.Stack = fmt.Sprintf("%+v", err)
		}
	}
	gRPCService := ""
	gRPCMethod := ""
	for _, field := range fields {
		switch field.Key {
		case KeyIPHeader:
			msg.IP = field.String
		case KeyRequestID:
			msg.ReqID = field.String
		case KeyHost:
			msg.Host = field.String
		case KeyUserAgent:
			msg.Useragent = field.String
		case KeyGRPCService:
			gRPCService = field.String
		case KeyGRPCMethod:
			gRPCMethod = field.String
		default:
			if _, ok := e.additionalFields[field.Key]; ok {
				msg.Additional[field.Key] = field.String
			}
		}
	}
	msg.URL = fmt.Sprintf("%s/%s", gRPCService, gRPCMethod)
	go func() {
		_ = e.client.SendMessage(msg)
	}()
	return nil
}

func (e errorBoosterLoggerZapCore) Sync() error {
	return nil
}
