package logging

import (
	"fmt"
	"net/url"
	"syscall"

	"github.com/getsentry/sentry-go"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/log/zap/logrotate"
)

type Config struct {
	Level                        string   `config:"logging-level"`
	SentryDSN                    string   `config:"logging-sentrydsn"`
	SentryLevel                  string   `config:"logging-sentrylevel"`
	ErrorBoosterLevel            string   `config:"logging-errorbooster-level"`
	ErrorBoosterAdditionalFields []string `config:"logging-errorbooster-additional-fields"`
}

var DefaultConfig = Config{
	Level:                        "info",
	ErrorBoosterLevel:            "error",
	ErrorBoosterAdditionalFields: nil,
}

// New constructs Qloud compatible logger
func New(config *Config) (*arczap.Logger, error) {
	return NewLogger(WithConfig(config), WithFormat(QloudFormat), WithSentry(config.SentryDSN != ""))
}

func NewDeploy(config *Config) (*arczap.Logger, error) {
	return NewLogger(WithConfig(config), WithFormat(DeployFormat), WithSentry(config.SentryDSN != ""))
}

func ConfigureSentry(sentryDSN string) error {
	return sentry.Init(sentry.ClientOptions{
		Dsn:              sentryDSN,
		Debug:            false,
		AttachStacktrace: true,
		Integrations: func(integrations []sentry.Integration) []sentry.Integration {
			var output []sentry.Integration
			for _, integration := range integrations {
				if integration.Name() == "ContextifyFrames" {
					continue
				}
				output = append(output, integration)
			}
			return output
		},
	})
}

func NewSentryLogger(level log.Level, dsn string) (*arczap.Logger, error) {
	err := ConfigureSentry(dsn)
	if err != nil {
		return nil, fmt.Errorf("cannot configure logging to sentry: %w", err)
	}
	return &arczap.Logger{L: zap.New(NewSentryLoggerZapCore(arczap.ZapifyLevel(level)))}, nil
}

func newFileLoggerByConfig(cfg *zap.Config, logPath string, opts ...zap.Option) (*arczap.Logger, error) {
	pathURI, err := url.ParseRequestURI(logPath)
	if err != nil {
		return nil, err
	}
	fileSink, err := logrotate.NewLogrotateSink(pathURI, syscall.SIGUSR1)
	if err != nil {
		return nil, err
	}
	encoder := zapcore.NewConsoleEncoder(cfg.EncoderConfig)
	core := zapcore.NewCore(encoder, fileSink, cfg.Level)
	return arczap.NewWithCore(core, opts...), nil
}

func newJSONFileLoggerByConfig(cfg *zap.Config, logPath string, opts ...zap.Option) (*arczap.Logger, error) {
	pathURI, err := url.ParseRequestURI(logPath)
	if err != nil {
		return nil, err
	}
	fileSink, err := logrotate.NewLogrotateSink(pathURI, syscall.SIGUSR1)
	if err != nil {
		return nil, err
	}
	encoder := zapcore.NewJSONEncoder(cfg.EncoderConfig)
	core := zapcore.NewCore(encoder, fileSink, cfg.Level)
	return arczap.NewWithCore(core, opts...), nil
}

func NewYtFileLogger(level log.Level, logPath string, opts ...zap.Option) (*arczap.Logger, error) {
	cfg := zap.Config{
		Level: zap.NewAtomicLevelAt(arczap.ZapifyLevel(level)),
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "",
			StacktraceKey:  "",
			TimeKey:        "",
			CallerKey:      "",
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.RFC3339TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}

	return newFileLoggerByConfig(&cfg, logPath, opts...)
}

func NewFileLogger(level log.Level, logPath string, opts ...zap.Option) (*arczap.Logger, error) {
	cfg := zap.Config{
		Level: zap.NewAtomicLevelAt(arczap.ZapifyLevel(level)),
		EncoderConfig: zapcore.EncoderConfig{
			MessageKey:     "msg",
			LevelKey:       "level",
			StacktraceKey:  "stackTrace",
			TimeKey:        "time",
			CallerKey:      "caller",
			EncodeLevel:    zapcore.CapitalLevelEncoder,
			EncodeTime:     zapcore.RFC3339TimeEncoder,
			EncodeDuration: zapcore.StringDurationEncoder,
			EncodeCaller:   zapcore.ShortCallerEncoder,
		},
	}

	return newFileLoggerByConfig(&cfg, logPath, opts...)
}

func NewJSONFileLogger(level log.Level, logPath string, opts ...zap.Option) (*arczap.Logger, error) {
	cfg := arczap.JSONConfig(level)
	return newJSONFileLoggerByConfig(&cfg, logPath, opts...)
}

func MergeLoggers(loggers ...*arczap.Logger) *arczap.Logger {
	var subCores []zapcore.Core
	for _, l := range loggers {
		subCores = append(subCores, l.L.Core())
	}
	core := zapcore.NewTee(subCores...)
	return arczap.NewWithCore(core)
}
