package logging

import (
	"fmt"
	"reflect"
	"runtime"
	"strings"
	"time"

	"github.com/getsentry/sentry-go"
	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/library/go/x/xruntime"
)

func NewSentryLoggerZapCore(level zapcore.Level) zapcore.Core {
	return &sentryLoggerZapCore{
		LevelEnabler: zap.NewAtomicLevelAt(level),
	}
}

type sentryLoggerZapCore struct {
	zapcore.LevelEnabler
	fields []zapcore.Field
}

func (s *sentryLoggerZapCore) With(fields []zapcore.Field) zapcore.Core {
	return &sentryLoggerZapCore{
		s.LevelEnabler,
		append(s.fields, fields...),
	}
}

func (s *sentryLoggerZapCore) Check(entry zapcore.Entry, checked *zapcore.CheckedEntry) *zapcore.CheckedEntry {
	if s.Enabled(entry.Level) {
		checked.AddCore(entry, s)
	}
	return checked
}

func (s *sentryLoggerZapCore) Write(entry zapcore.Entry, fields []zapcore.Field) error {
	event := sentry.NewEvent()

	err := findErrorInZapFields(fields)
	errorChain := unwrapChain(err)

	event.Extra = s.collectExtra(fields, errorChain)

	// dropping frames for logging
	stacktrace := filterFrames(sentry.NewStacktrace())

	exceptionizeEvent(event, errorChain, stacktrace, entry.Message)

	event.Threads = []sentry.Thread{{
		Stacktrace: stacktrace,
		Crashed:    entry.Level == zapcore.PanicLevel || entry.Level == zapcore.FatalLevel,
		Current:    true,
	}}

	event.Message = entry.Message
	event.Level = mapZapLevelToSentryLevel(entry.Level)
	event.Timestamp = entry.Time
	event.Logger = entry.LoggerName

	sentry.CaptureEvent(event)

	if entry.Level == zapcore.FatalLevel || entry.Level == zapcore.PanicLevel || entry.Level == zapcore.DPanicLevel {
		return s.Sync()
	}
	return nil
}

func (s *sentryLoggerZapCore) Sync() error {
	ok := sentry.Flush(60 * time.Second)
	if !ok {
		return xerrors.NewSentinel("sentry sync timeout")
	}
	return nil
}

func (s *sentryLoggerZapCore) collectExtra(fields []zapcore.Field, errorChain []error) map[string]interface{} {
	output := make(map[string]interface{})

	mobenc := zapcore.NewMapObjectEncoder()
	// fields from context
	for _, field := range s.fields {
		field.AddTo(mobenc)
	}
	// fields passed directly into logging function
	for _, field := range fields {
		field.AddTo(mobenc)
	}

	for k, v := range mobenc.Fields {
		output[k] = v
	}

	// fields from errors that implement Fields interface
	for i := len(errorChain) - 1; i >= 0; i-- {
		err := errorChain[i]
		switch err := err.(type) {
		case interface{ Fields() map[string]interface{} }:
			for k, v := range err.Fields() {
				output[k] = v
			}
		}
	}

	return output
}

func mapZapLevelToSentryLevel(level zapcore.Level) sentry.Level {
	switch level {
	case zapcore.DebugLevel:
		return sentry.LevelDebug
	case zapcore.InfoLevel:
		return sentry.LevelInfo
	case zapcore.WarnLevel:
		return sentry.LevelWarning
	case zapcore.ErrorLevel, zapcore.DPanicLevel, zapcore.PanicLevel:
		return sentry.LevelError
	case zapcore.FatalLevel:
		return sentry.LevelFatal
	}
	return sentry.LevelInfo
}

func unwrapChain(err error) []error {
	var errors []error
	for err != nil {
		errors = append(errors, err)
		switch previous := err.(type) {
		case interface{ Unwrap() error }:
			err = previous.Unwrap()
		case interface{ Cause() error }:
			err = previous.Cause()
		default:
			err = nil
		}
	}
	return errors
}

// exceptionizeEvent fill sentry event with data from error chain
func exceptionizeEvent(event *sentry.Event, errorChain []error, stacktrace *sentry.Stacktrace, message string) {
	if len(errorChain) == 0 {
		event.Exception = append(event.Exception, sentry.Exception{
			Value:      message,
			Type:       message,
			Stacktrace: stacktrace,
		})
		return
	}

	var customRootCause error // use to print a message in sentry issues list
	var customRootCauseType string

	for _, err := range errorChain {
		errType := reflect.TypeOf(err).String()
		event.Exception = append(event.Exception, sentry.Exception{
			Value:      err.Error(),
			Type:       errType,
			Stacktrace: extractStacktrace(err),
		})
		// try to skip generic xerrors errors
		if !strings.HasPrefix(errType, "*xerrors") {
			customRootCause = err
			customRootCauseType = fmt.Sprintf("%s: %s", message, errType)
		}
	}
	// if all errors in chain are from xerrors package, just use the first one
	if customRootCause == nil {
		customRootCause = errorChain[0]
		customRootCauseType = message
	}

	// event.Exception should be sorted such that the most recent error is last.
	reverse(event.Exception)

	event.Exception = append(event.Exception, sentry.Exception{
		Value:      customRootCause.Error(),
		Type:       customRootCauseType,
		Stacktrace: stacktrace,
	})
}

func extractStacktrace(err error) *sentry.Stacktrace {
	if err == nil {
		return nil
	}
	switch err := err.(type) {
	case interface{ StackTrace() *xruntime.StackTrace }:
		return xruntimeToSentryTrace(err.StackTrace())
	}
	return nil
}

func pcsToSentryTrace(pcs []uintptr) *sentry.Stacktrace {
	sentryFrames := extractFrames(pcs)
	if len(sentryFrames) == 0 {
		return nil
	}
	return &sentry.Stacktrace{
		Frames: sentryFrames,
	}
}

func xruntimeToSentryTrace(trace *xruntime.StackTrace) *sentry.Stacktrace {
	if trace == nil {
		return nil
	}
	frames := trace.Frames()
	if len(frames) == 0 {
		return nil
	}
	var pcs []uintptr
	for _, frame := range frames {
		pcs = append(pcs, frame.PC)
	}
	return pcsToSentryTrace(pcs)
}

func extractFrames(pcs []uintptr) []sentry.Frame {
	var frames []sentry.Frame
	callersFrames := runtime.CallersFrames(pcs)

	for {
		callerFrame, more := callersFrames.Next()

		frames = append([]sentry.Frame{
			sentry.NewFrame(callerFrame),
		}, frames...)

		if !more {
			break
		}
	}

	return frames
}

func filterFrames(stacktrace *sentry.Stacktrace) *sentry.Stacktrace {
	const dropFramesAfterModules = "a.yandex-team.ru/library/go/core/log/zap"

	for i, frame := range stacktrace.Frames {
		if frame.Module == dropFramesAfterModules {
			stacktrace.Frames = stacktrace.Frames[:i]
			break
		}
	}
	return stacktrace
}

// reverse reverses the slice a in place.
func reverse(a []sentry.Exception) {
	for i := len(a)/2 - 1; i >= 0; i-- {
		opp := len(a) - 1 - i
		a[i], a[opp] = a[opp], a[i]
	}
}
