package metrics

import (
	"sort"
	"strings"
	"sync"

	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/mock"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
)

type AppMetrics struct {
	mx                 sync.Mutex
	appMetricsRegistry metrics.Registry
	gaugeStore         map[string]map[string]metrics.Gauge
	counterStore       map[string]map[string]metrics.Counter
	timerStore         map[string]map[string]metrics.Timer
	registryStore      map[string]metrics.Registry
}

func NewAppMetrics(appMetricsRegistry metrics.Registry) *AppMetrics {
	return &AppMetrics{
		appMetricsRegistry: appMetricsRegistry,
		gaugeStore:         make(map[string]map[string]metrics.Gauge),
		counterStore:       make(map[string]map[string]metrics.Counter),
		timerStore:         make(map[string]map[string]metrics.Timer),
		registryStore:      make(map[string]metrics.Registry),
	}
}

func generateRegistryKey(prefix string, tags map[string]string) string {
	if tags == nil {
		return prefix
	}
	items := make([]string, len(tags))
	for key, value := range tags {
		items = append(items, key+"."+value)
	}
	sort.Strings(items)
	return prefix + "." + strings.Join(items, ":")
}

func (am *AppMetrics) GetOrCreateGauge(prefix string, tags map[string]string, gaugeName string) metrics.Gauge {
	key := generateRegistryKey(prefix, tags)
	am.mx.Lock()
	defer am.mx.Unlock()
	if _, found := am.gaugeStore[key]; !found {
		am.gaugeStore[key] = make(map[string]metrics.Gauge)
	}
	if gauge, found := am.gaugeStore[key][gaugeName]; found {
		return gauge
	}
	if _, found := am.registryStore[key]; !found {
		am.registryStore[key] = am.appMetricsRegistry.WithPrefix(prefix).WithTags(tags)
	}
	gauge := createGauge(am.registryStore[key], gaugeName)
	am.gaugeStore[key][gaugeName] = gauge
	return gauge
}

func (am *AppMetrics) GetOrCreateCounter(prefix string, tags map[string]string, counterName string) metrics.Counter {
	key := generateRegistryKey(prefix, tags)
	am.mx.Lock()
	defer am.mx.Unlock()
	if _, found := am.counterStore[key]; !found {
		am.counterStore[key] = make(map[string]metrics.Counter)
	}
	if counter, found := am.counterStore[key][counterName]; found {
		return counter
	}
	if _, found := am.registryStore[key]; !found {
		am.registryStore[key] = am.appMetricsRegistry.WithPrefix(prefix).WithTags(tags)
	}
	counter := createCounter(am.registryStore[key], counterName)
	am.counterStore[key][counterName] = counter
	return counter
}

func (am *AppMetrics) GetOrCreateHistogram(prefix string, tags map[string]string, timerName string, buckets metrics.DurationBuckets) metrics.Timer {
	key := generateRegistryKey(prefix, tags)
	am.mx.Lock()
	defer am.mx.Unlock()
	if _, found := am.timerStore[key]; !found {
		am.timerStore[key] = make(map[string]metrics.Timer)
	}
	if timer, found := am.timerStore[key][timerName]; found {
		return timer
	}
	if _, found := am.registryStore[key]; !found {
		am.registryStore[key] = am.appMetricsRegistry.WithPrefix(prefix).WithTags(tags)
	}
	timer := createHistogram(am.registryStore[key], timerName, buckets)
	am.timerStore[key][timerName] = timer
	return timer
}

func createGauge(appMetricsRegistry metrics.Registry, name string) metrics.Gauge {
	return appMetricsRegistry.Gauge(name)
}

func createHistogram(registry metrics.Registry, name string, durationBuckets metrics.DurationBuckets) metrics.Timer {
	requestTimingsHistogram := registry.DurationHistogram(
		name,
		durationBuckets,
	)
	solomon.Rated(requestTimingsHistogram)
	return requestTimingsHistogram
}

func createCounter(appMetricsRegistry metrics.Registry, name string) metrics.Counter {
	counter := appMetricsRegistry.Counter(name)
	solomon.Rated(counter)
	return counter
}

var globalAppMetrics = NewAppMetrics(mock.NewRegistry(mock.NewRegistryOpts()))

func SetGlobalAppMetrics(appMetrics *AppMetrics) {
	globalAppMetrics = appMetrics
}

func GlobalAppMetrics() *AppMetrics {
	return globalAppMetrics
}
