package server

import (
	"context"
	"errors"
	"net/http"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/library/go/core/log"
	coremetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/yandex/solomon/reporters/puller/httppuller"
)

var ErrorWrongRegistryType = errors.New("wrong registry type error")

type MetricsConfig struct {
	Addr                       string `config:"metrics-addr,required"`
	PerfMetricsPrefix          string
	PerfMetricsRefreshInterval time.Duration
}

var DefaultMetricsConfig = MetricsConfig{
	Addr:                       "[::]:9947",
	PerfMetricsRefreshInterval: 5 * time.Second,
	PerfMetricsPrefix:          "perf",
}

func RunMetricsHTTPServer(ctx context.Context, cfg MetricsConfig, logger log.Logger, registry coremetrics.Registry) error {
	r := chi.NewRouter()
	r.Use(middleware.Recoverer)
	sr, ok := registry.(*solomon.Registry)
	if !ok {
		return ErrorWrongRegistryType
	}
	handler := httppuller.NewHandler(sr)
	r.Get("/metrics/", handler.ServeHTTP)
	server := &http.Server{
		Addr:    cfg.Addr,
		Handler: r,
	}
	go func() {
		doneChan := ctx.Done()
		if doneChan != nil {
			<-doneChan
			logger.Info("Shutting down metrics server", log.String("address", cfg.Addr))
			_ = server.Shutdown(ctx)
		}
	}()
	err := server.ListenAndServe()
	if ctx.Err() == context.Canceled && err == http.ErrServerClosed {
		return nil // serverClosed is not an error if context is cancelled, we are just exiting
	} else {
		return err
	}
}
