package redislib

import (
	"context"
	"fmt"
	"time"

	"github.com/go-redis/redis/v8"

	"a.yandex-team.ru/library/go/core/metrics"
	tm "a.yandex-team.ru/travel/library/go/metrics"
)

type startKeyType struct{}

var startKey = startKeyType{}

type MetricsHook struct {
	prefix         string
	timingsBuckets metrics.DurationBuckets
}

type MetricsOption func(hook *MetricsHook)

func Prefix(prefix string) MetricsOption {
	return func(hook *MetricsHook) {
		hook.prefix = prefix
	}
}

func TimingsBuckets(timingsBuckets metrics.DurationBuckets) MetricsOption {
	return func(hook *MetricsHook) {
		hook.timingsBuckets = timingsBuckets
	}
}

func NewMetricsHook(opts ...MetricsOption) *MetricsHook {
	mh := MetricsHook{
		prefix:         "cache",
		timingsBuckets: nil,
	}
	for _, opt := range opts {
		opt(&mh)
	}
	return &mh
}

func (hook *MetricsHook) BeforeProcess(ctx context.Context, cmd redis.Cmder) (context.Context, error) {
	if hook.timingsBuckets != nil {
		ctx = context.WithValue(ctx, startKey, time.Now())
	}
	tm.GlobalAppMetrics().GetOrCreateCounter(fmt.Sprintf("%s-%s", hook.prefix, cmd.Name()), nil, "count").Inc()
	return ctx, nil
}

func (hook *MetricsHook) AfterProcess(ctx context.Context, cmd redis.Cmder) error {
	prefix := fmt.Sprintf("%s-%s", hook.prefix, cmd.Name())
	if hook.timingsBuckets != nil {
		if start, ok := ctx.Value(startKey).(time.Time); ok {
			duration := time.Since(start)
			tm.GlobalAppMetrics().GetOrCreateHistogram(prefix, nil, "duration", hook.timingsBuckets).RecordDuration(duration)
		}
	}

	err := cmd.Err()
	if err == nil {
		tm.GlobalAppMetrics().GetOrCreateCounter(prefix, nil, "success").Inc()
	} else if err == redis.Nil {
		tm.GlobalAppMetrics().GetOrCreateCounter(prefix, nil, "miss").Inc()
	} else {
		tm.GlobalAppMetrics().GetOrCreateCounter(prefix, nil, "error").Inc()
	}

	return nil
}

func (hook *MetricsHook) BeforeProcessPipeline(ctx context.Context, cmds []redis.Cmder) (context.Context, error) {
	return ctx, nil
}

func (hook *MetricsHook) AfterProcessPipeline(ctx context.Context, cmds []redis.Cmder) error {
	return nil
}
