package renderer

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/httputil/headers"
)

type Config struct {
	RendererURL string
	HTTPClient  *http.Client
}

type HTTPClient struct {
	backOffPolicyGetter func() backoff.BackOff
	httpClient          *http.Client
	config              Config
	requestTimeout      time.Duration
}

func NewHTTPClient(config Config, opts ...Option) *HTTPClient {
	c := &HTTPClient{config: config, httpClient: config.HTTPClient}
	defaultOpts := []Option{withDefaultBackOffPolicy, withDefaultRequestTimeout}
	for _, opt := range append(defaultOpts, opts...) {
		opt(c)
	}
	return c
}

type Block interface {
}

func (c *HTTPClient) Render(ctx context.Context, blocks []Block) (string, error) {
	blocksJSON, err := json.Marshal(blocks)
	if err != nil {
		return "", err
	}
	requestCtx, cancel := context.WithTimeout(ctx, c.requestTimeout)
	defer cancel()
	req, err := c.buildRequest(requestCtx, blocksJSON)
	if err != nil {
		return "", err
	}
	byteResponse, err := c.request(req)
	if err != nil {
		return "", RequestRendererError{err}
	}
	return string(byteResponse), nil
}

func (c *HTTPClient) RenderStructured(ctx context.Context, blocks []Block) (*StructuredHTML, error) {
	blocksJSON, err := json.Marshal(blocks)
	if err != nil {
		return nil, err
	}
	requestCtx, cancel := context.WithTimeout(ctx, c.requestTimeout)
	defer cancel()
	req, err := c.buildRequest(requestCtx, blocksJSON)
	if err != nil {
		return nil, err
	}
	byteResponse, err := c.request(req)
	if err != nil {
		return nil, RequestRendererError{err}
	}
	response := &StructuredHTML{}
	if err := json.Unmarshal(byteResponse, response); err != nil {
		return nil, InvalidResponseError{err}
	}
	return response, nil
}

func (c *HTTPClient) buildRequest(requestCtx context.Context, jsonData []byte) (*http.Request, error) {
	req, err := http.NewRequestWithContext(requestCtx, http.MethodPost, c.config.RendererURL, bytes.NewReader(jsonData))
	if err != nil {
		return nil, err
	}
	req.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	return req, nil
}

func (c *HTTPClient) request(req *http.Request) ([]byte, error) {
	var byteResponse []byte
	requestFunc := func() error {
		response, err := c.httpClient.Do(req)
		if err != nil {
			return RequestRendererError{err}
		}
		defer response.Body.Close()

		byteResponse, err = ioutil.ReadAll(response.Body)
		if err != nil {
			return RequestRendererError{err}
		}
		if response.StatusCode != http.StatusOK {
			err := fmt.Errorf("renderer answered with code %d: %s", response.StatusCode, string(byteResponse))
			if response.StatusCode == http.StatusBadRequest {
				return backoff.Permanent(err)
			}
			return RequestRendererError{err}
		}

		return nil
	}
	return byteResponse, c.retryRequest(requestFunc)
}

func (c *HTTPClient) retryRequest(request func() error) error {
	requestBackoff := c.backOffPolicyGetter()
	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}
