package resourcestorage

import (
	"context"
	"errors"
	"fmt"
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
)

type Iterable interface {
	Iter(ctx context.Context) <-chan proto.Message
}

type Dumper struct {
	logger      log.Logger
	container   Iterable
	resourceKey string
	writer      StorageWriter
	keepLast    int
}

func NewDumper(
	container Iterable, resourceKey string, writer StorageWriter, keepLast int, logger log.Logger,
) *Dumper {
	return &Dumper{
		logger:      logger,
		container:   container,
		resourceKey: resourceKey,
		writer:      writer,
		keepLast:    keepLast,
	}
}

func (d *Dumper) Dump() (int, error) {
	const errorMessageFormat = "Dumper.Dump fails: %w"
	err := d.writer.CreateVersion(d.resourceKey)
	defer d.writer.Close()
	if err != nil {
		return 0, fmt.Errorf(errorMessageFormat, err)
	}

	ctx, cancel := context.WithCancel(context.Background())
	defer cancel()
	n := 0
	for record := range d.container.Iter(ctx) {
		err = d.writer.Write(record)
		if err != nil {
			return 0, fmt.Errorf(errorMessageFormat, err)
		}
		n++
	}
	err = d.writer.Commit()
	if err != nil {
		return 0, fmt.Errorf(errorMessageFormat, err)
	}
	err = d.writer.CleanOldVersions(d.resourceKey, d.keepLast)
	if err != nil {
		return 0, fmt.Errorf(errorMessageFormat, err)
	}
	return n, nil
}

func (d *Dumper) RunPeriodic(period time.Duration, ctx context.Context) {
	go func() {
		ticker := time.NewTicker(period)
		for {
			select {
			case <-ticker.C:
				n, err := d.Dump()
				if err != nil {
					d.logger.Errorf("Dump.RunPeriodic fails for %s: %s", d.resourceKey, err.Error())
				} else {
					d.logger.Infof("Dump.RunPeriodic %d records dumped for %s", n, d.resourceKey)
				}
			case <-ctx.Done():
				return
			}
		}
	}()
}

type Extendable interface {
	Add(message proto.Message)
}

type Loader struct {
	logger      log.Logger
	resourceKey string
	reader      StorageReader
	sample      proto.Message
}

func NewLoader(
	sample proto.Message, resourceKey string, reader StorageReader, logger log.Logger,
) *Loader {
	return &Loader{
		logger:      logger,
		resourceKey: resourceKey,
		reader:      reader,
		sample:      sample,
	}
}

func (l *Loader) Load(container Extendable) (int, error) {
	const errorMessageFormat = "Loader.Load fails: %w"
	err := l.reader.Open(l.resourceKey)
	if err != nil {
		return 0, fmt.Errorf(errorMessageFormat, err)
	}
	defer l.reader.Close()
	n := 0
	for {
		record := proto.Clone(l.sample)
		err = l.reader.Read(record)
		if err != nil {
			if errors.Is(err, ErrStopIteration) {
				break
			}
			return n, fmt.Errorf(errorMessageFormat, err)
		}
		container.Add(record)
		n++
	}
	return n, nil
}
