package resourcestorage

import (
	"bytes"
	"crypto/md5"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path"
	"sort"
	"time"

	"github.com/golang/protobuf/proto"

	"a.yandex-team.ru/travel/library/go/dicts/base"
	ipb "a.yandex-team.ru/travel/proto/resourcestorage"
)

type LocalStorageWriter struct {
	rootPath        string
	started         bool
	keepLast        int
	metaPath        string
	resourceMeta    *ipb.ResourceMeta
	versionMeta     *ipb.ResourceVersionMeta
	generator       *base.BytesGenerator
	generatorWriter *os.File
}

func NewLocalStorageWriter(rootPath string) *LocalStorageWriter {
	return &LocalStorageWriter{
		rootPath: rootPath,
		started:  false,
	}
}

func (lsw *LocalStorageWriter) CreateVersion(key string) error {
	const errorMessage = "LocalStorageWriter.CreateVersion fails"
	if lsw.started {
		return fmt.Errorf("%s: already created", errorMessage)
	}
	version := makeVersionName()
	versionDir := path.Join(lsw.rootPath, key, version)
	if _, err := os.Stat(versionDir); !os.IsNotExist(err) {
		return fmt.Errorf("%s: directory %s exists", errorMessage, versionDir)
	}
	err := os.MkdirAll(versionDir, os.ModePerm)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	lsw.resourceMeta = &ipb.ResourceMeta{
		Key:       key,
		Version:   version,
		Timestamp: time.Now().Unix(),
	}
	lsw.versionMeta = &ipb.ResourceVersionMeta{
		Version:   version,
		Timestamp: lsw.resourceMeta.Timestamp,
	}
	dataPath := path.Join(lsw.rootPath, lsw.resourceMeta.Key, lsw.versionMeta.Version, versionDataFn)
	lsw.generatorWriter, err = os.Create(dataPath)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	lsw.generator, _ = base.BuildGeneratorForWriter(lsw.generatorWriter)
	lsw.started = true
	return nil
}

func (lsw *LocalStorageWriter) Write(message proto.Message) error {
	const errorMessage = "LocalStorageWriter.Write fails"
	if !lsw.started {
		return fmt.Errorf("%s: not started", errorMessage)
	}

	messageBytes, err := proto.Marshal(message)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	err = lsw.generator.Write(messageBytes)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	return nil
}

func (lsw *LocalStorageWriter) Commit() error {
	const errorMessage = "LocalStorageWriter.Commit fails"
	if !lsw.started {
		return fmt.Errorf("%s: not started", errorMessage)
	}
	defer lsw.Close()

	err := lsw.generatorWriter.Close()
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	lsw.versionMeta.Md5 = lsw.generator.HashSum()

	versionMetaFd, err := os.Create(path.Join(lsw.rootPath, lsw.resourceMeta.Key, lsw.versionMeta.Version, versionMetaFn))
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	versionMetaBytes, err := proto.Marshal(lsw.versionMeta)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	_, err = versionMetaFd.Write(versionMetaBytes)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	err = versionMetaFd.Close()
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}

	resourceMetaFd, err := os.Create(path.Join(lsw.rootPath, lsw.resourceMeta.Key, resourceMetaFn))
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	defer func() { _ = resourceMetaFd.Close() }()
	resourceMetaBytes, err := proto.Marshal(lsw.resourceMeta)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	_, err = resourceMetaFd.Write(resourceMetaBytes)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	err = resourceMetaFd.Close()
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	return nil
}

func (lsw *LocalStorageWriter) deleteVersion(key string, version string) error {
	err := os.RemoveAll(path.Join(lsw.rootPath, key, version))
	if err != nil {
		return fmt.Errorf("LocalStorageWriter.deleteVersion fails: %w", err)
	}
	return nil
}

func (lsw *LocalStorageWriter) CleanOldVersions(key string, keepLast int) error {
	const errorMessageFormat = "LocalStorageWriter.CleanOldVersions fails: %w"
	lsr := NewLocalStorageReader(lsw.rootPath)
	resourceMeta, err := lsr.loadResourceMeta(key)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	versions, err := lsr.getVersions(key)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	sort.Strings(versions)
	for i := 0; i < len(versions)-keepLast; i++ {
		if versions[i] == resourceMeta.Version {
			continue
		}
		e := lsw.deleteVersion(key, versions[i])
		if e != nil {
			err = e
		}
	}
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

func (lsw *LocalStorageWriter) Close() {
	lsw.started = false
}

type LocalStorageReader struct {
	rootPath         string
	opened           bool
	currentMessageID int
	resourceMeta     *ipb.ResourceMeta
	versionMeta      *ipb.ResourceVersionMeta
	iterator         *base.BytesIterator
	reader           *os.File
}

func NewLocalStorageReader(rootPath string) *LocalStorageReader {
	return &LocalStorageReader{
		rootPath:     rootPath,
		opened:       false,
		resourceMeta: &ipb.ResourceMeta{},
		versionMeta:  &ipb.ResourceVersionMeta{},
	}
}

func (lsr *LocalStorageReader) Open(key string) error {
	const errorMessageFormat = "LocalStorageReader.Open fails: %w"
	resourceMeta, err := lsr.loadResourceMeta(key)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	err = lsr.openVersion(key, resourceMeta.Version, true)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

func (lsr *LocalStorageReader) Read(msg proto.Message) error {
	const errorMessage = "LocalStorageReader.Read fails: %w"
	if !lsr.opened {
		return fmt.Errorf("%s: not opened", errorMessage)
	}

	msgBytes, err := lsr.iterator.Next()
	if err != nil {
		if errors.Is(err, base.ErrStopIteration) {
			return ErrStopIteration
		}
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	err = proto.Unmarshal(msgBytes, msg)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	return nil
}

func (lsr *LocalStorageReader) Close() {
	lsr.opened = false
	_ = lsr.reader.Close()
}

func (lsr *LocalStorageReader) GetTimestamp(key string) (time.Time, error) {
	resourceMeta, err := lsr.loadResourceMeta(key)
	if err != nil {
		return time.Time{}, fmt.Errorf("LocalStorageReader.GetTimestamp fails: %w", err)
	}
	return time.Unix(resourceMeta.Timestamp, 0), nil
}

func (lsr *LocalStorageReader) loadResourceMeta(key string) (*ipb.ResourceMeta, error) {
	const errorMessageFormat = "LocalStorageReader.loadResourceMeta fails: %w"
	resourceMetaBytes, err := ioutil.ReadFile(path.Join(lsr.rootPath, key, resourceMetaFn))
	if err != nil {
		return nil, fmt.Errorf(errorMessageFormat, err)
	}
	resourceMeta := ipb.ResourceMeta{}
	err = proto.Unmarshal(resourceMetaBytes, &resourceMeta)
	if err != nil {
		return nil, fmt.Errorf(errorMessageFormat, err)
	}
	return &resourceMeta, nil
}

func (lsr *LocalStorageReader) checkMD5(fN string, checkSum []byte) bool {
	fd, err := os.Open(fN)
	if err != nil {
		return false
	}
	defer func() { _ = fd.Close() }()
	hash := md5.New()
	if _, err := io.Copy(hash, fd); err != nil {
		return false
	}
	s := hash.Sum(nil)
	return bytes.Equal(s[:], checkSum)
}

func (lsr *LocalStorageReader) openVersion(key string, version string, checkControlSum bool) error {
	const errorMessage = "LocalStorageReader.openVersion fails"
	if lsr.opened {
		return fmt.Errorf("%s: already opened", errorMessage)
	}

	resourceMeta, err := lsr.loadResourceMeta(key)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	lsr.resourceMeta = resourceMeta
	versionMetaBytes, err := ioutil.ReadFile(path.Join(lsr.rootPath, key, version, versionMetaFn))
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}

	lsr.versionMeta = &ipb.ResourceVersionMeta{}
	err = proto.Unmarshal(versionMetaBytes, lsr.versionMeta)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	dataFn := path.Join(lsr.rootPath, key, version, versionDataFn)
	if checkControlSum && !lsr.checkMD5(dataFn, lsr.versionMeta.Md5) {
		return fmt.Errorf("%s: bad checksum", errorMessage)
	}
	lsr.reader, err = os.Open(dataFn)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	lsr.iterator, err = base.BuildIteratorFromReader(lsr.reader)
	if err != nil {
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	lsr.opened = true
	return nil
}

func (lsr *LocalStorageReader) getVersions(key string) ([]string, error) {
	dirs, err := ioutil.ReadDir(path.Join(lsr.rootPath, key))
	if err != nil {
		return []string{}, fmt.Errorf("LocalStorageReader.getVersions fails: %w", err)
	}
	var versions []string
	tryingReader := NewLocalStorageReader(lsr.rootPath)
	for _, dir := range dirs {
		if !dir.IsDir() {
			continue
		}
		version := dir.Name()
		err = tryingReader.openVersion(key, version, false)
		tryingReader.Close()
		if err != nil {
			continue
		}
		versions = append(versions, version)
	}
	return versions, nil
}
