package resourcestorage

import (
	"errors"
	"fmt"

	"github.com/golang/protobuf/proto"
)

type ResourceWriterManager struct {
	writers []StorageWriter
}

func NewResourceWriterManager(writers ...StorageWriter) *ResourceWriterManager {
	return &ResourceWriterManager{
		writers: writers,
	}
}

func (rwm *ResourceWriterManager) CreateVersion(key string) error {
	for _, writer := range rwm.writers {
		err := writer.CreateVersion(key)
		if err != nil {
			return fmt.Errorf("ResourceWriterManager.Start fails: %w", err)
		}
	}
	return nil
}

func (rwm *ResourceWriterManager) Write(msg proto.Message) error {
	for _, writer := range rwm.writers {
		err := writer.Write(msg)
		if err != nil {
			return fmt.Errorf("ResourceWriterManager.WriteChunk fails: %w", err)
		}
	}
	return nil
}

func (rwm *ResourceWriterManager) Commit() error {
	for _, writer := range rwm.writers {
		err := writer.Commit()
		if err != nil {
			return fmt.Errorf("ResourceWriterManager.Commit fails: %w", err)
		}
	}
	return nil
}

func (rwm *ResourceWriterManager) Close() {
	for _, writer := range rwm.writers {
		writer.Close()
	}
}

func (rwm *ResourceWriterManager) CleanOldVersions(key string, keepLast int) error {
	for _, writer := range rwm.writers {
		err := writer.CleanOldVersions(key, keepLast)
		if err != nil {
			return fmt.Errorf("ResourceWriterManager.CleanOldVersions fails: %w", err)
		}
	}
	return nil
}

type ResourceReaderManager struct {
	readers []StorageReader
}

func NewResourceReaderManager(readers ...StorageReader) *ResourceReaderManager {
	return &ResourceReaderManager{
		readers: readers,
	}
}

func (rrm *ResourceReaderManager) Open(key string) error {
	if len(rrm.readers) > 0 {
		err := rrm.readers[0].Open(key)
		if err != nil {
			return fmt.Errorf("ResourceReaderManager.OpenLastVersion fails: %w", err)
		}
		return nil
	}
	return fmt.Errorf("ResourceReaderManager.OpenLastVersion fails: no readers provided")
}

func (rrm *ResourceReaderManager) Read(msg proto.Message) error {
	if len(rrm.readers) > 0 {
		err := rrm.readers[0].Read(msg)
		if err != nil {
			if errors.Is(err, ErrStopIteration) {
				return err
			}
			return fmt.Errorf("ResourceReaderManager.ReadMessage fails: %w", err)
		}
		return nil
	}
	return fmt.Errorf("ResourceReaderManager.ReadMessage fails: no readers provided")
}

func (rrm *ResourceReaderManager) Close() {
	for _, reader := range rrm.readers {
		reader.Close()
	}
}
