// This is storage for protobuf models
// Supported local file system and MDS-S3 backends
// Write and Read methods works in streaming mode without big memory usage
//
// Files structure:
//	Root (local directory or s3 bucket)
//		ResourceKey
//			Version1
//				data.pb
//				version.pb
//			Version2
//				data.pb
//				version.pb
//			resource.pb
//
package resourcestorage

import (
	"errors"
	"fmt"
	"math/rand"
	"time"

	"github.com/golang/protobuf/proto"
)

const (
	resourceMetaFn = "resource.pb"
	versionMetaFn  = "version.pb"
	versionDataFn  = "data.pb"
	localCacheSize = 2
)

var ErrStopIteration = errors.New("no more entries")

type StorageWriter interface {
	CreateVersion(key string) error
	Write(msg proto.Message) error
	Commit() error
	Close()
	CleanOldVersions(key string, keepLast int) error
}

type StorageReader interface {
	Open(key string) error
	Read(msg proto.Message) error
	Close()
	GetTimestamp(key string) (time.Time, error)
}

func makeVersionName() string {
	return fmt.Sprintf("%X-%04X", time.Now().UnixNano(), rand.Intn(0xFFFF))
}
