package resourcestorage

import (
	"errors"
	"fmt"
	"os"
	"path"
	"sort"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/golang/protobuf/proto"

	ipb "a.yandex-team.ru/travel/proto/resourcestorage"
)

type S3StorageConfig struct {
	LocalStoragePath  string `config:"storage-localstoragepath,required"`
	S3StorageEndpoint string `config:"storage-s3storageendpoint,required"`
	S3StorageBucket   string `config:"storage-s3storagebucket,required"`
	S3StorageRegion   string `config:"storage-s3storageregion,required"`
}

var DefaultS3StorageConfig = S3StorageConfig{
	LocalStoragePath: "/ephemeral/api",
	S3StorageRegion:  "yandex",
}

var MockedS3StorageConfig = S3StorageConfig{
	LocalStoragePath: path.Join(os.TempDir(), "resource_storage"),
}

type S3SessionProvider struct {
	endpoint  string
	region    string
	accessKey string
	secret    string
}

func NewS3SessionProvider(endpoint string, region string, accessKey string, secret string) *S3SessionProvider {
	return &S3SessionProvider{
		endpoint:  endpoint,
		region:    region,
		accessKey: accessKey,
		secret:    secret,
	}
}

func (sp *S3SessionProvider) GetSession() (*session.Session, error) {
	return session.NewSession(&aws.Config{
		Endpoint:    &sp.endpoint,
		Region:      &sp.region,
		Credentials: credentials.NewStaticCredentials(sp.accessKey, sp.secret, ""),
	})
}

type S3StorageWriter struct {
	bucket          string
	lsw             *LocalStorageWriter
	sessionProvider *S3SessionProvider
}

func NewS3StorageWriter(cfg S3StorageConfig, accessKey string, secret string) StorageWriter {
	if cfg == MockedS3StorageConfig {
		_ = os.RemoveAll(cfg.LocalStoragePath)
		return NewLocalStorageWriter(cfg.LocalStoragePath)
	}
	return &S3StorageWriter{
		sessionProvider: NewS3SessionProvider(
			cfg.S3StorageEndpoint, cfg.S3StorageRegion, accessKey, secret),
		lsw:    NewLocalStorageWriter(cfg.LocalStoragePath),
		bucket: cfg.S3StorageBucket,
	}
}

func (ssw *S3StorageWriter) CreateVersion(key string) error {
	return ssw.lsw.CreateVersion(key)
}

func (ssw *S3StorageWriter) Write(message proto.Message) error {
	return ssw.lsw.Write(message)
}

func (ssw *S3StorageWriter) Commit() error {
	errorMessageFormat := "S3StorageWriter.Commit fails: %w"
	err := ssw.lsw.Commit()
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}

	sess, err := ssw.sessionProvider.GetSession()
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	uploader := s3manager.NewUploader(sess)
	for _, pathKey := range [][]string{
		{
			path.Join(ssw.lsw.rootPath, ssw.lsw.resourceMeta.Key, ssw.lsw.versionMeta.Version, versionDataFn),
			path.Join(ssw.lsw.resourceMeta.Key, ssw.lsw.versionMeta.Version, versionDataFn),
		},
		{
			path.Join(ssw.lsw.rootPath, ssw.lsw.resourceMeta.Key, ssw.lsw.versionMeta.Version, versionMetaFn),
			path.Join(ssw.lsw.resourceMeta.Key, ssw.lsw.versionMeta.Version, versionMetaFn),
		},
		{
			path.Join(ssw.lsw.rootPath, ssw.lsw.resourceMeta.Key, resourceMetaFn),
			path.Join(ssw.lsw.resourceMeta.Key, resourceMetaFn),
		},
	} {
		reader, err := os.Open(pathKey[0])
		if err != nil {
			return fmt.Errorf(errorMessageFormat, err)
		}
		_, err = uploader.Upload(&s3manager.UploadInput{
			Bucket: &ssw.bucket,
			Key:    &pathKey[1],
			Body:   reader,
		})
		if err != nil {
			return fmt.Errorf(errorMessageFormat, err)
		}
	}
	return nil
}

func (ssw *S3StorageWriter) Close() {
	ssw.lsw.Close()
}

func (ssw *S3StorageWriter) CleanOldVersions(key string, keepLast int) error {
	const errorMessageFormat = "S3StorageWriter.CleanOldVersions fails: %w"
	_ = ssw.lsw.CleanOldVersions(key, localCacheSize)
	ssr := NewS3StorageReaderBySessionProvider(ssw.lsw.rootPath, ssw.bucket, ssw.sessionProvider)
	resourceMeta := ipb.ResourceMeta{}
	err := ssr.loadResourceToProto(path.Join(key, resourceMetaFn), &resourceMeta)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	versions, err := ssr.getVersions(key)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	sort.Strings(versions)
	for i := 0; i < len(versions)-keepLast; i++ {
		if versions[i] == resourceMeta.Version {
			continue
		}
		e := ssw.deleteVersion(key, versions[i])
		if e != nil {
			err = e
		}
	}
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

func (ssw *S3StorageWriter) deleteVersion(key string, version string) error {
	errorMessageFormat := "S3StorageWriter.deleteVersion fails: %w"
	_ = ssw.lsw.deleteVersion(key, version)
	sess, err := ssw.sessionProvider.GetSession()
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	s3Client := s3.New(sess)
	versionMetaKey := path.Join(key, version, versionMetaFn)
	versionDataKey := path.Join(key, version, versionDataFn)
	_, err = s3Client.DeleteObjects(&s3.DeleteObjectsInput{
		Bucket: &ssw.bucket,
		Delete: &s3.Delete{
			Objects: []*s3.ObjectIdentifier{
				{Key: &versionMetaKey},
				{Key: &versionDataKey},
			},
		},
	})
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

type S3StorageReader struct {
	bucket          string
	lsr             *LocalStorageReader
	sessionProvider *S3SessionProvider
}

func NewS3StorageReader(cfg S3StorageConfig, accessKey string, secret string) StorageReader {
	if cfg == MockedS3StorageConfig {
		return NewLocalStorageReader(cfg.LocalStoragePath)
	}
	sessionProvider := NewS3SessionProvider(
		cfg.S3StorageEndpoint, cfg.S3StorageRegion, accessKey, secret)
	return NewS3StorageReaderBySessionProvider(cfg.LocalStoragePath, cfg.S3StorageBucket, sessionProvider)
}

func NewS3StorageReaderBySessionProvider(
	rootPath string, bucket string, sessionProvider *S3SessionProvider,
) *S3StorageReader {
	lsr := NewLocalStorageReader(rootPath)
	return &S3StorageReader{
		sessionProvider: sessionProvider,
		bucket:          bucket,
		lsr:             lsr,
	}
}

func (ssr *S3StorageReader) Open(key string) error {
	errorMessageFormat := "S3StorageReader.Open fails: %w"
	resourceMeta := ipb.ResourceMeta{}
	err := ssr.loadResourceToProto(path.Join(key, resourceMetaFn), &resourceMeta)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	err = ssr.openVersion(key, resourceMeta.Version)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

func (ssr *S3StorageReader) Read(msg proto.Message) error {
	errorMessage := "S3StorageReader.Read fails"
	err := ssr.lsr.Read(msg)
	if err != nil {
		if errors.Is(err, ErrStopIteration) {
			return ErrStopIteration
		}
		return fmt.Errorf("%s: %w", errorMessage, err)
	}
	return nil
}

func (ssr *S3StorageReader) Close() {
	ssr.lsr.Close()
}

func (ssr *S3StorageReader) GetTimestamp(key string) (time.Time, error) {
	resourceMeta := ipb.ResourceMeta{}
	err := ssr.loadResourceToProto(path.Join(key, resourceMetaFn), &resourceMeta)
	if err != nil {
		return time.Time{}, fmt.Errorf("S3StorageReader.GetTimestamp fails: %w", err)
	}
	return time.Unix(resourceMeta.Timestamp, 0), nil
}

func (ssr *S3StorageReader) loadResourceToProto(key string, message proto.Message) error {
	errorMessageFormat := "S3StorageReader.loadResourceToProto fails: %w"
	sess, err := ssr.sessionProvider.GetSession()
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	var buffer []byte
	writeAtBuffer := aws.NewWriteAtBuffer(buffer)
	downloader := s3manager.NewDownloader(sess)
	downloader.Concurrency = 1
	_, err = downloader.Download(writeAtBuffer, &s3.GetObjectInput{
		Bucket: &ssr.bucket,
		Key:    &key,
	})
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	err = proto.Unmarshal(writeAtBuffer.Bytes(), message)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

func (ssr *S3StorageReader) openVersion(key string, version string) error {
	errorMessageFormat := "S3StorageReader.openVersion fails: %w"
	// first try to open from local cache
	err := ssr.lsr.openVersion(key, version, true)
	if err == nil {
		return nil
	}
	err = os.MkdirAll(path.Join(ssr.lsr.rootPath, key, version), os.ModePerm)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	lsw := NewLocalStorageWriter(ssr.lsr.rootPath)
	_ = lsw.CleanOldVersions(key, localCacheSize-1)
	sess, err := ssr.sessionProvider.GetSession()
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	downloader := s3manager.NewDownloader(sess)
	for _, fN := range []string{
		path.Join(key, version, versionDataFn),
		path.Join(key, version, versionMetaFn),
		path.Join(key, resourceMetaFn),
	} {
		fD, err := os.Create(path.Join(ssr.lsr.rootPath, fN))
		if err != nil {
			return fmt.Errorf(errorMessageFormat, err)
		}
		_, err = downloader.Download(fD, &s3.GetObjectInput{
			Bucket: &ssr.bucket,
			Key:    &fN,
		})
		if err != nil {
			return fmt.Errorf(errorMessageFormat, err)
		}
	}
	err = ssr.lsr.openVersion(key, version, true)
	if err != nil {
		return fmt.Errorf(errorMessageFormat, err)
	}
	return nil
}

func listObjectKeys(s3Client *s3.S3, bucket string, prefix string) ([]string, error) {
	objects, err := s3Client.ListObjects(&s3.ListObjectsInput{
		Bucket: &bucket,
		Prefix: &prefix,
	})
	if err != nil {
		return nil, err
	}
	keys := make([]string, len(objects.Contents))
	for i, o := range objects.Contents {
		keys[i] = *o.Key
	}
	return keys, nil
}

func (ssr *S3StorageReader) getVersions(key string) ([]string, error) {
	errorMessageFormat := "S3StorageReader.getVersions fails: %w"
	sess, err := ssr.sessionProvider.GetSession()
	if err != nil {
		return nil, fmt.Errorf(errorMessageFormat, err)
	}
	s3Client := s3.New(sess)
	keys, err := listObjectKeys(s3Client, ssr.bucket, key)
	if err != nil {
		return nil, fmt.Errorf(errorMessageFormat, err)
	}
	var versions []string
	for _, key := range keys {
		if !strings.HasSuffix(key, versionMetaFn) {
			continue
		}
		versionMeta := &ipb.ResourceVersionMeta{}
		err = ssr.loadResourceToProto(key, versionMeta)
		if err != nil {
			continue
		}
		versions = append(versions, versionMeta.Version)
	}
	return versions, nil
}
