package sandboxplanner

import (
	"encoding/json"
	"fmt"
	"os"
	"strings"
)

const ArgReplacesEnvKey = "ARG_REPLACES"

type replaceParamsMapType map[string]string

func GetReplacedArgsFromEnv(args []string, replacesEnvKey ...string) ([]string, error) {
	if len(args) == 0 {
		args = os.Args[1:]
	}

	if len(args) == 0 {
		return args, nil
	}

	replaceParamsMap, err := getReplaceParamsMap(replacesEnvKey...)
	if err != nil {
		return nil, fmt.Errorf("sandboxplanner.GetReplacedArgsFromEnv: %v", err)
	}

	for i, arg := range args {
		args[i] = replaceArg(arg, replaceParamsMap)
	}
	return args, nil
}

func ReplaceArgsFromEnv(replacesEnvKey ...string) error {
	if len(os.Args) == 1 {
		return nil
	}

	replaceParamsMap, err := getReplaceParamsMap(replacesEnvKey...)
	if err != nil {
		return fmt.Errorf("sandboxplanner.ReplaceArgsFromEnv: %v", err)
	}
	for i := 1; i < len(os.Args); i++ {
		os.Args[i] = replaceArg(os.Args[i], replaceParamsMap)
	}
	return nil
}

func getReplaceParamsMap(replacesEnvKey ...string) (replaceParamsMapType, error) {
	envKey := ArgReplacesEnvKey
	if len(replacesEnvKey) > 0 {
		envKey = replacesEnvKey[0]
	}

	argReplaces, ok := os.LookupEnv(envKey)
	if !ok {
		return nil, nil
	}

	replaceParamsMap := make(replaceParamsMapType)
	if err := json.Unmarshal([]byte(argReplaces), &replaceParamsMap); err != nil {
		return nil, fmt.Errorf("sandboxplanner.getReplaceParamsMap: %v", err)
	}
	return replaceParamsMap, nil
}

func replaceArg(arg string, replaceParamsMap replaceParamsMapType) string {
	for k, v := range replaceParamsMap {
		arg = strings.ReplaceAll(arg, k, v)
	}
	return arg
}
