package sender

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"strconv"
	"strings"
	"time"

	"github.com/cenkalti/backoff/v4"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/headers"
)

const (
	senderOKStatus    = "OK"
	senderErrorStatus = "ERROR"
)

type HTTPClient struct {
	httpClient          *http.Client
	logger              log.Logger
	config              Config
	requestTimeout      time.Duration
	backOffPolicyGetter func() backoff.BackOff
}

func NewHTTPClient(logger log.Logger, config Config, opts ...Option) *HTTPClient {
	c := &HTTPClient{
		config:     config,
		httpClient: config.HTTPClient,
		logger:     logger,
	}
	defaultOpts := []Option{withDefaultBackOffPolicy, withDefaultRequestTimeout}
	for _, opt := range append(defaultOpts, opts...) {
		opt(c)
	}
	return c
}

func (c *HTTPClient) SendTransactional(ctx context.Context, sendRequest TransactionalRequest) ([]byte, error) {
	requestCtx, cancel := context.WithTimeout(ctx, c.requestTimeout)
	defer cancel()
	req, err := c.buildTransactionalRequest(requestCtx, sendRequest)
	if err != nil {
		return nil, err
	}
	return c.doRequest(req)
}

func (c *HTTPClient) Unsubscribe(ctx context.Context, request UnsubscribeListRequest) error {
	return c.updateUnsubscribeList(ctx, request, http.MethodPut)
}

func (c *HTTPClient) Subscribe(ctx context.Context, request UnsubscribeListRequest) error {
	return c.updateUnsubscribeList(ctx, request, http.MethodDelete)
}

func (c *HTTPClient) updateUnsubscribeList(ctx context.Context, request UnsubscribeListRequest, httpMethod string) error {
	requestCtx, cancel := context.WithTimeout(ctx, c.requestTimeout)
	defer cancel()
	req, err := c.buildUnsubscribeListRequest(requestCtx, request, httpMethod)
	if err != nil {
		return err
	}
	_, err = c.doRequest(req)
	return err
}

func (c *HTTPClient) doRequest(req *http.Request) (responseBytes []byte, err error) {
	requestFunc := func() error {
		response, err := c.httpClient.Do(req)
		if err != nil {
			return RequestSenderError{err}
		}
		defer response.Body.Close()

		responseBytes, err = ioutil.ReadAll(response.Body)
		if err != nil {
			return RequestSenderError{err}
		}
		if response.StatusCode >= http.StatusInternalServerError {
			return RequestSenderError{fmt.Errorf("sender responded with code %d: %s", response.StatusCode, string(responseBytes))}
		}
		if err = c.parseResponse(responseBytes); err != nil {
			return err
		}
		return nil
	}
	if err = c.retryRequest(requestFunc); err != nil {
		return nil, err
	}
	return responseBytes, nil
}

func (c *HTTPClient) retryRequest(request func() error) error {
	requestBackoff := c.backOffPolicyGetter()
	requestBackoff.Reset()
	return backoff.Retry(request, requestBackoff)
}

func (c HTTPClient) parseResponse(data []byte) error {
	responseParsed := responseRoot{}
	err := json.Unmarshal(data, &responseParsed)
	if err != nil {
		return ResponseParsingError{fmt.Errorf("got error while unmarshal data (%s): %v", string(data), err)}
	}

	status := strings.ToUpper(responseParsed.Result.Status)
	if status == senderErrorStatus && len(responseParsed.Result.Error.Email) > 0 {
		return backoff.Permanent(NewInvalidEmailError(responseParsed.Result.Error.Email))
	}

	if status != senderOKStatus {
		return fmt.Errorf("sender responded with error status: %s", string(data))
	}
	return nil
}

func (c *HTTPClient) buildTransactionalRequest(ctx context.Context, sendRequest TransactionalRequest) (*http.Request, error) {
	sendURL := c.getTransactionalSendingURL(sendRequest.CampaignSlug)

	bodyReader, err := c.getTransactionalRequestBody(sendRequest)
	if err != nil {
		return nil, InvalidRequestError{err}
	}

	req, err := http.NewRequestWithContext(ctx, http.MethodPost, sendURL, bodyReader)
	if err != nil {
		return nil, RequestSenderError{err}
	}
	req.Header.Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	req.SetBasicAuth(c.config.AuthKey, "")
	return req, nil
}

func (c *HTTPClient) getTransactionalSendingURL(campaignSlug string) string {
	return fmt.Sprintf(
		"%s/api/0/%s/transactional/%s/send",
		c.config.SenderURL,
		c.config.Account,
		campaignSlug,
	)
}

func (c *HTTPClient) getTransactionalRequestBody(sendRequest TransactionalRequest) (io.Reader, error) {
	params := map[string]interface{}{
		"async":       strconv.FormatBool(sendRequest.SendAsync),
		"to_email":    sendRequest.ToEmail,
		"for_testing": strconv.FormatBool(c.config.AllowInactiveCampaigns),
		"args":        sendRequest.Args,
		"headers":     sendRequest.Headers,
	}
	paramsBytes, err := json.Marshal(params)
	if err != nil {
		return nil, fmt.Errorf("couldn't encode request params %v to json: %w", params, err)
	}
	return bytes.NewReader(paramsBytes), nil
}

func (c *HTTPClient) getUnsubscribeListURL(unsubscribeListSlug string) string {
	return fmt.Sprintf(
		"%s/api/0/%s/unsubscribe/list/%s",
		c.config.SenderURL,
		c.config.Account,
		unsubscribeListSlug,
	)
}

func (c *HTTPClient) buildUnsubscribeListRequest(
	ctx context.Context,
	request UnsubscribeListRequest,
	httpMethod string,
) (*http.Request, error) {
	unsubscribeListURL := c.getUnsubscribeListURL(request.UnsubscribeListSlug)
	req, err := http.NewRequestWithContext(ctx, httpMethod, unsubscribeListURL, nil)
	if err != nil {
		return nil, RequestSenderError{err}
	}
	query := req.URL.Query()
	query.Add("email", request.Email)
	req.URL.RawQuery = query.Encode()
	req.SetBasicAuth(c.config.AuthKey, "")
	return req, nil
}
