package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/library/go/grpcutil/service/servingstatus"
	"github.com/jonboulle/clockwork"
	ghealth "google.golang.org/grpc/health"

	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
	pbapi "a.yandex-team.ru/travel/library/go/service_template/api"
	"a.yandex-team.ru/travel/library/go/service_template/internal/app"
	echohandler "a.yandex-team.ru/travel/library/go/service_template/internal/echo/handler"
	"a.yandex-team.ru/travel/library/go/service_template/internal/healthswitch"
	healthhandler "a.yandex-team.ru/travel/library/go/service_template/internal/healthswitch/handler"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "service_template"

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		return
	}

	logger, err := logging.New(&app.Cfg.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		return
	}

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)

	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Error("tracer close error:", log.Error(err))
		}

		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
		ctxCancel()
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(app.Cfg.Tvm.WhiteList)
	if app.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewQloudClient()
		if err != nil {
			fmt.Println("failed to create tvm client:", err)
		}
		// At 100 rps we had 0.1 rps errors (exceeded timeout 0.5 seconds) while check service-ticket
		// in tvm-daemon in deploy using standard client. See RASPTICKETS-20639 for details
		// So use cached-tvm-client
		// You can set cache options via cachedtvm.WithCheckServiceTicket and cachedtvm.WithCheckUserTicket
		tvmClient, err = cachedtvm.NewClient(tvmClient)
		if err != nil {
			fmt.Println("failed to create cached tvm client:", err)
		}
	}

	rootRegistry := metrics.NewRegistryWithQloudTags()
	// in order to make it possible to collect arbitrary application metrics through metrics.GlobalAppMetrics
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	// bootstrapping business logic
	httpEchoHandler := echohandler.NewHTTPEchoHandler(logger)
	grpcEchoHandler := echohandler.NewGRPCEchosHandler(logger)

	healthStatus := healthswitch.NewSwitch()
	healthSwitchHandler := healthhandler.NewHTTPHealthSwitchHandler(healthStatus)

	httpRouteBuilders := []httpserver.RouteBuilder{
		httpEchoHandler.GetRouteBuilder(),
		healthSwitchHandler.GetRouteBuilder(),
	}

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		grpcEchoHandler.GetServiceRegisterer(),
	}

	// Example of working with GRPC health server
	healthServer := ghealth.NewServer()
	// The health-checking logic of a specific application should be implemented below
	servingStatusService := servingstatus.NewService(
		logger,
		serviceName,
		healthServer.SetServingStatus,
		app.Cfg.HealthCheck.UpdateInterval,
		clockwork.NewRealClock(),
	).Requires(
		func() bool {
			return healthStatus.GetHealth()
		},
	)
	servingStatusService.MonitorServingStatus(ctx)

	grpcGateway := grpcgateway.NewGateway(&app.Cfg.GrpcGateway, grpcgateway.NewService(serviceName, "/echo", app.Cfg.Grpc.Addr, pbapi.RegisterEchoServiceV1HandlerFromEndpoint, nil))

	// run server
	wg := sync.WaitGroup{}
	wg.Add(4)

	go func() {
		defer wg.Done()
		err = httpserver.RunHTTPServer(context.Background(), app.Cfg.HTTP, httpRouteBuilders, logger, tvmClient, tvmAllowedIds, rootRegistry.WithPrefix("http"))
		if err != nil {
			logger.Fatal("Error while starting HTTP server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		server := grpcserver.NewDefaultGrpcServerBuilder(
			app.Cfg.Grpc,
			grpcServiceRegisterers,
			logger,
			tvmClient,
			tvmAllowedIds,
			rootRegistry.WithPrefix("grpc"),
		).WithHealthServer(healthServer).Build()
		err = server.Run(context.Background())
		if err != nil {
			logger.Fatal("Error while starting Grpc server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = metricserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	go func() {
		defer wg.Done()
		err = grpcGateway.Run(ctx)
		if err != nil {
			logger.Error("GRPC-gateway server closed", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
