package handler

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/httputil"
	"a.yandex-team.ru/travel/library/go/service_template/internal/echo"
	"encoding/json"
	"fmt"
	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"
	"net/http"
	"net/url"
	"strconv"
)

type HTTPEchoHandler struct {
	logger log.Logger
}

type response struct {
	Response string `json:"response"`
}

func NewHTTPEchoHandler(l log.Logger) *HTTPEchoHandler {
	return &HTTPEchoHandler{logger: l}
}

func (h *HTTPEchoHandler) Echo(w http.ResponseWriter, r *http.Request) {
	span, _ := opentracing.StartSpanFromContext(r.Context(), "internal.echo.handler.HTTPEchoHandler: Echo")
	defer span.Finish()

	params, err := url.ParseQuery(r.URL.RawQuery)
	if err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	mode := params.Get("echo_mode")
	echoMode, err := strconv.Atoi(mode)
	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		httputil.HandleError(fmt.Errorf("internal error"), http.StatusBadRequest, w)
		return
	}
	query := params.Get("query")
	result, err := echo.ProcessQuery(echo.Mode(echoMode), query)
	if err != nil {
		h.logger.Error("Internal error", log.Error(err))
		httputil.HandleError(fmt.Errorf("internal error"), http.StatusInternalServerError, w)
		return
	}

	resultBytes, err := json.Marshal(response{Response: result})
	if err != nil {
		httputil.HandleError(err, http.StatusInternalServerError, w)
		return
	}
	w.WriteHeader(http.StatusOK)
	w.Header().Set("Content-Type", "application/json")
	_, _ = w.Write(resultBytes)
}

func (h *HTTPEchoHandler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/echo", h.Echo)
	}
}
