package messagecollector

import (
	"context"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/library/go/syncutil"
)

type Collector struct {
	logger log.Logger

	routes              []*route
	countEnabledSources int
}

func NewCollector(logger log.Logger) *Collector {
	return &Collector{logger: logger.WithName("Collector")}
}

func (c *Collector) Route(enabled bool, source Source, dispatcher MessageDispatcher, options ...RouteOption) *Collector {
	if enabled {
		c.countEnabledSources++
	}

	route := newRoute(
		enabled,
		source,
		dispatcher,
		options...,
	)
	c.routes = append(
		c.routes,
		route,
	)
	return c
}

func (c Collector) Run(ctx context.Context) {
	wg := syncutil.WaitGroup{}

	for _, route := range c.routes {
		source := route.source
		dispatcher := BuildDispatcher(
			route.baseDispatcher,
			route.limiters,
			route.backoffPolicy,
			route.dispatcherWrappers...,
		)
		if route.enabled {
			wg.Go(
				func() {
					c.logger.Infof("Collecting from %s source started", source.Name())
					source.Run(ctx, dispatcher)
				},
			)
		} else {
			c.logger.Infof("Collecting from %s is disabled", source.Name())
		}
	}

	c.logger.Info("Started")
	if c.countEnabledSources == 0 {
		c.logger.Info("all collectors are disabled so reader will just idle")
		wg.Add(1)
	}
	wg.Wait()
}

func (c Collector) Ready() bool {
	for _, rs := range c.routes {
		if rs.enabled && !rs.source.Ready() {
			return false
		}
	}
	return true
}
