package messagecollector

import (
	"github.com/cenkalti/backoff/v4"
)

type DispatcherWrapper func(MessageDispatcher) MessageDispatcher

func BuildDispatcher(
	dispatcher MessageDispatcher,
	limiters []Limiter,
	backoffPolicy backoff.BackOff,
	wrappers ...DispatcherWrapper,
) MessageDispatcher {
	for _, limiter := range limiters {
		dispatcher = NewLimitingDispatcher(dispatcher, limiter)
	}
	if backoffPolicy != nil {
		dispatcher = NewRetryingDispatcher(dispatcher, backoffPolicy)
	}
	for _, wrapper := range wrappers {
		dispatcher = wrapper(dispatcher)
	}
	return dispatcher
}

type LimitingDispatcher struct {
	dispatcher MessageDispatcher
	limiter    Limiter
}

func NewLimitingDispatcher(
	dispatcher MessageDispatcher,
	limiter Limiter,
) MessageDispatcher {
	return &LimitingDispatcher{
		dispatcher: dispatcher,
		limiter:    limiter,
	}
}

func (d LimitingDispatcher) Dispatch(message []byte) error {
	d.limiter.BeforeDispatch()
	defer d.limiter.AfterDispatch()
	return d.dispatcher.Dispatch(message)
}

type RetryingDispatcher struct {
	dispatcher    MessageDispatcher
	backoffPolicy backoff.BackOff
}

func NewRetryingDispatcher(
	dispatcher MessageDispatcher,
	backoffPolicy backoff.BackOff,
) MessageDispatcher {
	return &RetryingDispatcher{
		dispatcher:    dispatcher,
		backoffPolicy: backoffPolicy,
	}
}

func (d RetryingDispatcher) Dispatch(message []byte) error {
	if d.backoffPolicy == nil {
		return d.dispatcher.Dispatch(message)
	}

	return backoff.Retry(
		func() error {
			return d.dispatcher.Dispatch(message)
		},
		d.backoffPolicy,
	)
}
