package messagecollector

import "time"

type RPSLimiter struct {
	ticker *time.Ticker
}

func NewRPSLimiter(rpsLimit float32) *RPSLimiter {
	return &RPSLimiter{ticker: buildTicker(rpsLimit)}
}

func (l RPSLimiter) BeforeDispatch() {
	if l.ticker != nil {
		<-l.ticker.C
	}
}

func (l RPSLimiter) AfterDispatch() {}

func buildTicker(rpsLimit float32) *time.Ticker {
	if rpsLimit <= 0 {
		return nil
	}
	return time.NewTicker(time.Second * 1000 / time.Duration(rpsLimit*1000))
}

type InflightLimiter struct {
	limiter chan struct{}
}

func NewInflightLimiter(maxInflight uint) *InflightLimiter {
	return &InflightLimiter{limiter: buildLimiter(maxInflight)}
}

func (l InflightLimiter) BeforeDispatch() {
	if l.limiter != nil {
		l.limiter <- struct{}{}
	}
}

func (l InflightLimiter) AfterDispatch() {
	if l.limiter != nil {
		<-l.limiter
	}
}

func buildLimiter(maxInflight uint) chan struct{} {
	if maxInflight <= 0 {
		return nil
	}
	return make(chan struct{}, maxInflight)
}
