package sandbox

import (
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"path/filepath"
	"time"

	"github.com/cenkalti/backoff/v4"
)

const BaseURL = "https://sandbox.yandex-team.ru/api/v1.0"

type method string

const resourceMethod method = "resource"

type Client interface {
	GetResources(ResourceQuery) (*ResourceResponse, error)
	DownloadAllResourceBytes(ResourceItems) ([]byte, error)
	DownloadResource(item ResourceItems) (io.ReadCloser, error)
}

type HTTPClient struct {
	baseURL    string
	httpClient *http.Client
	backOff    backoff.BackOff
	oauthToken string
}

func NewHTTPClient(baseURL string, httpClient *http.Client, oauthToken string) Client {
	b := &backoff.ExponentialBackOff{
		InitialInterval:     backoff.DefaultInitialInterval,
		RandomizationFactor: backoff.DefaultRandomizationFactor,
		Multiplier:          backoff.DefaultMultiplier,
		MaxInterval:         time.Second,
		MaxElapsedTime:      3 * time.Second,
		Clock:               backoff.SystemClock,
		Stop:                backoff.Stop,
	}
	b.Reset()

	return &HTTPClient{
		baseURL:    baseURL,
		httpClient: httpClient,
		backOff:    b,
		oauthToken: oauthToken,
	}
}

func (client *HTTPClient) GetResources(query ResourceQuery) (*ResourceResponse, error) {
	if err := query.validate(); err != nil {
		return nil, err
	}

	requestURL, err := client.buildURL(resourceMethod, &query)
	if err != nil {
		return nil, err
	}

	response, err := client.get(requestURL)
	if response != nil {
		defer response.Body.Close()
	}
	if err != nil {
		return nil, err
	}

	if 400 <= response.StatusCode {
		return nil, fmt.Errorf("sandbox answered with an error: %v", response.StatusCode)
	}

	resourceResponse := &ResourceResponse{}

	err = json.NewDecoder(response.Body).Decode(resourceResponse)
	if err != nil {
		return nil, fmt.Errorf("couldn't decode sandbox answer: %+v", err)
	}
	return resourceResponse, err
}

func (client *HTTPClient) buildURL(relPath method, query Query) (requestURL string, err error) {
	baseURL, err := url.Parse(client.baseURL)
	if err != nil {
		return "", fmt.Errorf("bad sandbox base url: %+v. An error occured while parsing: %+v", client.baseURL, err)
	}

	baseURL.RawQuery = query.ToURLValues().Encode()
	baseURL.Path = filepath.Join(baseURL.Path, string(relPath))
	return baseURL.String(), nil
}

func (client *HTTPClient) DownloadAllResourceBytes(item ResourceItems) ([]byte, error) {
	resourceURL := item.HTTP.Proxy

	if resourceURL == "" {
		for _, link := range item.HTTP.Links {
			if link != "" {
				resourceURL = link
				break
			}
		}
	}

	if resourceURL == "" {
		return nil, fmt.Errorf("unable to find a link to download resource")
	}

	response, err := client.get(resourceURL)
	if response != nil {
		defer response.Body.Close()
	}
	if err != nil {
		return nil, fmt.Errorf("an error occured while download resource: %+v", err)
	}

	body, err := ioutil.ReadAll(response.Body)
	if err != nil {
		return nil, fmt.Errorf("an error occured while download resource: %+v", err)
	}
	return body, nil
}

func (client *HTTPClient) DownloadResource(item ResourceItems) (io.ReadCloser, error) {
	resourceURL := item.HTTP.Proxy

	if resourceURL == "" {
		for _, link := range item.HTTP.Links {
			if link != "" {
				resourceURL = link
				break
			}
		}
	}

	if resourceURL == "" {
		return nil, fmt.Errorf("unable to find a link to download resource")
	}

	response, err := client.get(resourceURL)
	if err != nil {
		return nil, fmt.Errorf("an error occured while download resource: %+v", err)
	}
	return response.Body, err
}

func (client *HTTPClient) get(requestURL string) (response *http.Response, err error) {
	err = backoff.RetryNotify(
		func() error {
			request, err := http.NewRequest(http.MethodGet, requestURL, nil)
			if err != nil {
				return err
			}
			if len(client.oauthToken) > 0 {
				request.Header.Add("Authorization", fmt.Sprintf("OAuth %s", client.oauthToken))
			}
			response, err = client.httpClient.Do(request)

			if err != nil {
				return err
			}

			if 500 <= response.StatusCode && response.StatusCode <= 600 {
				err = fmt.Errorf("sandbox answered with an error: %v", response.StatusCode)
			}

			return err
		},

		client.backOff,
		func(err error, duration time.Duration) {
			if response != nil {
				_ = response.Body.Close()
			}
		},
	)
	return
}
