package tanker

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"mime/multipart"
	"net/http"
	"net/url"
)

const (
	downloadKeysetPath = "keysets/tjson/"
	replaceKeysetPath  = "keysets/replace/"
)

type Client struct {
	baseURL string
	token   string
}

func NewClient(baseURL string, token string) *Client {
	return &Client{baseURL: baseURL, token: token}
}

func (c *Client) DownloadKeyset(branchID, projectID, keysetID string) (Keyset, error) {
	request, err := c.buildKeysetRequest(http.MethodGet, downloadKeysetPath, branchID, projectID, keysetID)
	if err != nil {
		return nil, err
	}
	// TODO: use https://a.yandex-team.ru/arc_vcs/travel/library/go/httputil/client/
	resp, err := http.DefaultClient.Do(request)
	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("tanker: bad response code=%s", resp.Status)
	}
	if resp != nil {
		defer resp.Body.Close()
	}
	if err != nil {
		return nil, err
	}
	project := Project{}
	err = json.NewDecoder(resp.Body).Decode(&project)
	if err != nil {
		return nil, err
	}
	keysetInfo, ok := project.KeysetInfos[keysetID]
	if !ok {
		return nil, fmt.Errorf("there is no keyset in branch %s with keyset-id %s in project %s", branchID, keysetID, projectID)
	}
	return keysetInfo.Keys, nil
}

func (c *Client) ReplaceKeyset(branchID, projectID, keysetID, format string, keyset Keyset) error {
	baseURL, _ := url.Parse(c.baseURL)
	baseURL.Path = replaceKeysetPath
	body := &bytes.Buffer{}
	writer := multipart.NewWriter(body)
	part, err := writer.CreateFormFile("file", "@data.file")
	if err != nil {
		return err
	}
	project := Project{KeysetInfos: map[string]*KeysetInfo{keysetID: {Keys: keyset}}}
	serializedProject, _ := json.Marshal(project)
	_, err = io.Copy(part, bytes.NewBuffer(serializedProject))
	if err != nil {
		return err
	}
	for key, val := range map[string]string{"project-id": projectID, "keyset-id": keysetID, "branch-id": branchID, "format": format} {
		_ = writer.WriteField(key, val)
	}
	err = writer.Close()
	if err != nil {
		return err
	}

	request, err := http.NewRequest(http.MethodPost, baseURL.String(), body)

	if err != nil {
		return err
	}
	request.Header.Set("Content-Type", writer.FormDataContentType())
	request.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.token))
	resp, err := http.DefaultClient.Do(request)
	if resp != nil {
		defer resp.Body.Close()
	}
	return err
}

func (c *Client) buildKeysetRequest(method, requestPath, branchID, projectID, keysetID string) (*http.Request, error) {
	baseURL, _ := url.Parse(c.baseURL)
	baseURL.Path = requestPath
	baseURL.RawQuery = url.Values{"project-id": {projectID}, "keyset-id": {keysetID}, "branch-id": {branchID}}.Encode()
	request, err := http.NewRequest(method, baseURL.String(), nil)
	if err != nil {
		return nil, err
	}
	request.Header.Add("Authorization", fmt.Sprintf("OAuth %s", c.token))
	return request, nil
}
