package next

import (
	"fmt"

	"golang.org/x/text/feature/plural"
	"golang.org/x/text/language"
	"golang.org/x/text/message"
	"golang.org/x/text/message/catalog"

	tankerpb "a.yandex-team.ru/travel/proto/tanker"
)

type KeySet struct {
	c      catalog.Catalog
	keySet *tankerpb.KeySet
}

func NewKeySet(keySet *tankerpb.KeySet) (*KeySet, error) {
	res := &KeySet{
		keySet: keySet,
	}
	cat := catalog.NewBuilder()
	for key, keyData := range keySet.Keys {
		for lang, translation := range keyData.Translations {
			payload := translation.Payload
			langTag, err := language.Parse(lang)
			if err != nil {
				return nil, err
			}
			var m catalog.Message
			if langTag == language.Russian {
				m = plural.Selectf(1, "",
					plural.One, payload.SingularForm,
					plural.Few, payload.FewForm,
					plural.Many, payload.ManyForm,
					plural.Other, payload.OtherForm,
				)
			} else if langTag == language.English {
				m = plural.Selectf(1, "",
					plural.One, payload.SingularForm,
					plural.Other, payload.OtherForm,
				)
			} else {
				// add supported plural forms for supported languages
				return nil, fmt.Errorf("tanker: language %s is not supported", lang)
			}
			err = cat.Set(langTag, key, m)
			if err != nil {
				return nil, err
			}
		}
	}
	res.c = cat
	return res, nil
}

func (k *KeySet) GetSingular(key, language string) (string, error) {
	payload, err := getPayload(k.keySet, key, language)
	if err != nil {
		return "", err
	}
	return payload.SingularForm, nil
}

func (k *KeySet) GetPlural(key, lang string, count int) (string, error) {
	_, err := getPayload(k.keySet, key, lang)
	if err != nil {
		return "", err
	}
	langTag, err := language.Parse(lang)
	if err != nil {
		return "", err
	}
	p := message.NewPrinter(langTag, message.Catalog(k.c))
	return p.Sprintf(key, count), nil
}

func (k *KeySet) GetKeys() []string {
	res := make([]string, len(k.keySet.Keys))
	i := 0
	for key := range k.keySet.Keys {
		res[i] = key
		i++
	}
	return res
}

func getPayload(keySet *tankerpb.KeySet, key, language string) (*tankerpb.TranslationPayload, error) {
	keyData, ok := keySet.Keys[key]
	if !ok {
		return nil, NewKeyNotFoundError(key)
	}
	translation, ok := keyData.Translations[language]
	if !ok {
		return nil, NewNoTranslationError(language, key)
	}
	return translation.Payload, nil
}
