package tvm

import (
	"net/http"
	"strconv"

	"a.yandex-team.ru/library/go/core/log"
	tvmutil "a.yandex-team.ru/library/go/httputil/middleware/tvm"
	yatvm "a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"google.golang.org/grpc"
)

type TvmHelperConfig struct {
	SelfID                    uint32   `yaml:"self-id"`
	EnabledServiceTicketCheck bool     `yaml:"enabled-service-ticket-check"`
	WhiteList                 []uint32 `yaml:"white-list"`
}

type TvmHelper struct {
	config      *TvmHelperConfig
	l           log.Logger
	tvm         yatvm.Client
	whiteList   []yatvm.ClientID
	initialized bool
}

func NewQloudTvmHelper(logger log.Logger, config *TvmHelperConfig) TvmHelper {
	h := TvmHelper{
		config:    config,
		l:         logger,
		whiteList: TvmClientIDFromInt(config.WhiteList),
	}
	var clientOptions []tvmtool.Option
	if config.SelfID != 0 {
		clientOptions = append(clientOptions, tvmtool.WithSrc(strconv.Itoa(int(config.SelfID))))
	}
	client, err := tvmtool.NewQloudClient(clientOptions...)
	if err != nil {
		logger.Error("create qloud tvm client failed:", log.Error(err))
		h.initialized = false
	} else {
		h.tvm = client
		h.initialized = true
	}
	return h
}

func NewDeployTvmHelper(logger log.Logger, config *TvmHelperConfig) TvmHelper {
	h := TvmHelper{
		config:    config,
		l:         logger,
		whiteList: TvmClientIDFromInt(config.WhiteList),
	}
	var clientOptions []tvmtool.Option
	if config.SelfID != 0 {
		clientOptions = append(clientOptions, tvmtool.WithSrc(strconv.Itoa(int(config.SelfID))))
	}
	client, err := tvmtool.NewDeployClient(clientOptions...)
	if err != nil {
		logger.Error("create deploy tvm client failed:", log.Error(err))
		h.initialized = false
	} else {
		h.tvm = client
		h.initialized = true
	}
	return h
}

func (h *TvmHelper) CheckServiceTicketHTTPMiddleware() func(next http.Handler) http.Handler {
	if h.config.EnabledServiceTicketCheck && h.initialized {
		return tvmutil.CheckServiceTicket(h.tvm, tvmutil.WithAllowedClients(h.whiteList), tvmutil.WithLogger(h.l.Structured()))
	}
	return nil
}

func (h *TvmHelper) CheckServiceTicketGRPCInterceptor() grpc.UnaryServerInterceptor {
	if h.config.EnabledServiceTicketCheck && h.initialized {
		return CheckServiceTicketInterceptor(h.tvm, WithAllowedClients(h.whiteList), WithLogger(h.l))
	}
	return nil
}

func (h *TvmHelper) GRPCClientInterceptor(dst uint32) grpc.UnaryClientInterceptor {
	if h.initialized {
		return ClientInterceptor(h.tvm, ClientWithLogger(h.l), ClientWithDst(dst))
	}
	return nil
}

func TvmClientIDFromInt(clientIds []uint32) []yatvm.ClientID {
	tvmAllowed := make([]yatvm.ClientID, len(clientIds))
	for i, c := range clientIds {
		tvmAllowed[i] = yatvm.ClientID(c)
	}
	return tvmAllowed
}
