package unifiedagent

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"sync"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	uapb "a.yandex-team.ru/logbroker/unified_agent/plugins/grpc_input/proto"
)

type ClientConfig struct {
	Enabled  bool   `config:"unifiedagent-client-enabled"`
	Endpoint string `config:"unifiedagent-client-endpoint,required"`
}

var DefaultClientConfig = ClientConfig{
	Enabled:  true,
	Endpoint: "127.0.0.1:16302",
}

type Client interface {
	Send(message proto.Message, meta []*uapb.Request_MessageMetaItem) error
	SendJSON(message interface{}, meta []*uapb.Request_MessageMetaItem) error
	GetAck() error
	Close() error
}

type GrpcClient struct {
	cfg         *ClientConfig
	grpcConn    *grpc.ClientConn
	grpcSession uapb.UnifiedAgentService_SessionClient
	seqNo       uint64
	sessionID   string
	logger      log.Logger
	mutex       sync.Mutex
}

func NewGrpcClient(cfg *ClientConfig, logger log.Logger, exSessionID *string) (Client, error) {
	const funcName = "unifiedagent.NewGrpcClient"
	if !cfg.Enabled {
		logger.Infof("%s: disabled", funcName)
		return &GrpcClient{
			cfg:    cfg,
			logger: logger,
		}, nil
	}

	opts := []grpc.DialOption{
		grpc.WithInsecure(),
	}

	conn, err := grpc.Dial(cfg.Endpoint, opts...)
	if err != nil {
		return nil, fmt.Errorf("%s: failed to init grpc connection: %w", funcName, err)
	}

	client := uapb.NewUnifiedAgentServiceClient(conn)
	session, err := client.Session(context.Background(), []grpc.CallOption{}...)
	if err != nil {
		return nil, fmt.Errorf("%s: failed to init grpc session: %w", funcName, err)
	}

	reconnectSession := ""
	if exSessionID != nil {
		reconnectSession = *exSessionID
	}

	request := &uapb.Request{
		Request: &uapb.Request_Initialize_{
			Initialize: &uapb.Request_Initialize{
				SessionId: reconnectSession,
			},
		},
	}

	logger.Infof("%s: connection established, send init message", funcName)

	err = session.Send(request)
	if err != nil {
		return nil, fmt.Errorf("%s: failed to send init request: %w", funcName, err)
	}

	resp, err := session.Recv()
	if err != nil {
		return nil, fmt.Errorf("%s: failed to send init response: %w", funcName, err)
	}

	logger.Infof("%s: server init response %s", funcName, resp.String())
	logger.Infof("%s: connection state %d", funcName, conn.GetState())

	return &GrpcClient{
		cfg:         cfg,
		grpcConn:    conn,
		grpcSession: session,
		seqNo:       resp.GetInitialized().GetLastSeqNo() + 1,
		sessionID:   resp.GetInitialized().GetSessionId(),
		logger:      logger,
		mutex:       sync.Mutex{},
	}, nil
}

func (a *GrpcClient) Send(message proto.Message, meta []*uapb.Request_MessageMetaItem) error {
	const funcName = "GrpcClient.Send"
	if !a.Enabled() {
		return nil
	}

	payload, err := proto.Marshal(message)
	if err != nil {
		return fmt.Errorf("%s: %w", funcName, err)
	}
	return a.send(payload, meta, funcName)
}

func (a *GrpcClient) SendJSON(message interface{}, meta []*uapb.Request_MessageMetaItem) error {
	const funcName = "GrpcClient.SendJSON"
	if !a.Enabled() {
		return nil
	}
	payload, err := json.Marshal(message)
	if err != nil {
		return fmt.Errorf("%s: %w", funcName, err)
	}
	return a.send(payload, meta, funcName)
}

func (a *GrpcClient) send(payload []byte, meta []*uapb.Request_MessageMetaItem, funcName string) error {
	a.mutex.Lock()
	defer a.mutex.Unlock()

	request := a.buildRequest(payload, meta)

	a.seqNo++
	err := a.grpcSession.Send(request)
	if err != nil {
		a.logger.Errorf("%s: failed to send grpc request: %s", funcName, err.Error())
		if err == io.EOF {
			a.logger.Errorf("%s: channel closed by server, restart grpc connection. Reconnecting...", funcName)
			recErr := a.reconnect()
			if recErr != nil {
				a.logger.Errorf("%s: failed to restart grpc connection: %s", funcName, recErr.Error())
			}
		} else {
			a.logger.Errorf("%s: network problems, resetting grpc client retry backoff", funcName)
			a.grpcConn.ResetConnectBackoff()
		}
		return err
	}

	err = a.GetAck()
	if err != nil {
		return fmt.Errorf("%s: cannot get Ack: %w", funcName, err)
	}
	return nil
}

func (a *GrpcClient) GetAck() error {
	const funcName = "GrpcClient.GetAck"
	if !a.Enabled() {
		return nil
	}

	resp, err := a.grpcSession.Recv()
	if err != nil {
		a.logger.Errorf("%s: failed to get grpc ack: %s", funcName, err.Error())
		if err == io.EOF {
			a.logger.Errorf("%s: channel closed by server, restart grpc connection. Reconnecting...", funcName)
			recErr := a.reconnect()
			if recErr != nil {
				a.logger.Errorf("%s: failed to restart grpc connection: %s", funcName, recErr.Error())
			}
		}
		return err
	}

	if resp.GetAck() == nil {
		return fmt.Errorf("%s: response is not ack", funcName)
	}
	return nil
}

func (a *GrpcClient) Close() error {
	if !a.cfg.Enabled {
		return nil
	}
	if err := a.grpcSession.CloseSend(); err != nil {
		return err
	}
	if _, err := a.grpcSession.Recv(); err != nil && !errors.Is(err, io.EOF) {
		return err
	}
	return a.grpcConn.Close()
}

func (a *GrpcClient) Enabled() bool {
	return a.cfg.Enabled
}

func (a *GrpcClient) reconnect() error {
	const funcName = "GrpcClient.reconnect"
	a.logger.Infof("%s: reconnecting %d", funcName, a.grpcConn.GetState())

	err := a.grpcConn.Close()
	if err != nil {
		a.logger.Errorf("%s: failed to close existing connection: %s", funcName, err.Error())
	}

	newCl, err := NewGrpcClient(a.cfg, a.logger, &a.sessionID)
	if err != nil {
		return fmt.Errorf("%s: failed to reconnect: %w", funcName, err)
	}

	a.seqNo = newCl.(*GrpcClient).seqNo
	a.grpcConn = newCl.(*GrpcClient).grpcConn
	a.grpcSession = newCl.(*GrpcClient).grpcSession

	return nil
}

func (a *GrpcClient) buildRequest(payload []byte, meta []*uapb.Request_MessageMetaItem) *uapb.Request {
	return &uapb.Request{
		Request: &uapb.Request_DataBatch_{
			DataBatch: &uapb.Request_DataBatch{
				SeqNo:     []uint64{a.seqNo},
				Timestamp: []uint64{uint64(time.Now().Unix())},
				Payload:   [][]byte{payload},
				Meta:      meta,
			},
		},
	}
}
