package vault

import (
	"context"
	"fmt"
	"os"
	"time"

	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

type YavSecretsResolver struct {
	client *httpyav.Client
	cache  map[string]map[string]string
}

func NewYavSecretsResolver() *YavSecretsResolver {
	c, _ := httpyav.NewClient(httpyav.WithOAuthToken(os.Getenv("TRAVEL_VAULT_OAUTH_TOKEN")))
	r := &YavSecretsResolver{
		client: c,
		cache:  make(map[string]map[string]string),
	}
	return r
}

func NewYavSecretsResolverFromEnv(envName string) *YavSecretsResolver {
	c, _ := httpyav.NewClient(httpyav.WithOAuthToken(os.Getenv(envName)))
	r := &YavSecretsResolver{
		client: c,
		cache:  make(map[string]map[string]string),
	}
	return r
}

func NewYavSecretsResolverFromToken(tokenValue string) *YavSecretsResolver {
	c, _ := httpyav.NewClient(httpyav.WithOAuthToken(tokenValue))
	r := &YavSecretsResolver{
		client: c,
		cache:  make(map[string]map[string]string),
	}
	return r
}

func (r *YavSecretsResolver) GetSecretValue(secUID string, key string) (string, error) {
	if _, ok := r.cache[secUID]; ok {
		return r.cache[secUID][key], nil
	}
	ctx, cancel := context.WithTimeout(context.Background(), 15*time.Second)
	defer cancel()
	secret, err := r.client.GetVersion(ctx, secUID)
	if err != nil {
		return "", err
	}
	if secret.Response.Status != "ok" {
		return "", fmt.Errorf("unexpected yav response %v", secret.Response)
	}
	secValue := ""
	secFound := false
	r.cache[secUID] = make(map[string]string, len(secret.Version.Values))
	for _, v := range secret.Version.Values {
		if v.Key == key {
			secValue = v.Value
			secFound = true
		}
		r.cache[secUID][v.Key] = v.Value
	}
	if !secFound {
		return "", fmt.Errorf("secret with key %v not found", key)
	}
	return secValue, nil
}
