package ru.yandex.travel.integration.balance;

import java.lang.reflect.Array;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpApiParsingException;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.integration.balance.model.BillingClient;
import ru.yandex.travel.integration.balance.model.BillingContract;
import ru.yandex.travel.integration.balance.model.BillingPerson;
import ru.yandex.travel.integration.balance.model.BillingUrPerson;
import ru.yandex.travel.integration.balance.requests.BillingClientContractsRequest;
import ru.yandex.travel.integration.balance.requests.BillingPersonRequest;
import ru.yandex.travel.integration.balance.responses.BillingCheckRUBankAccountResponse;
import ru.yandex.travel.integration.balance.responses.BillingCreateClientResponse;
import ru.yandex.travel.integration.balance.responses.BillingCreateContractResponse;
import ru.yandex.travel.integration.balance.xmlrpc.XmlRpcConverters;
import ru.yandex.travel.integration.balance.xmlrpc.XmlRpcHelper;
import ru.yandex.travel.integration.balance.xmlrpc.XmlRpcRequest;
import ru.yandex.travel.integration.balance.xmlrpc.XmlRpcRequestXmlConverter;
import ru.yandex.travel.integration.balance.xmlrpc.XmlRpcResponseXmlConverter;
import ru.yandex.travel.integration.balance.xmlrpc.XmlRpcRuntimeException;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
public class BillingApiClient extends HttpApiClientBase {
    public static final String BILLING_TIMEZONE_STRING = "Europe/Moscow";
    public static final ZoneId BILLING_TIMEZONE = ZoneId.of(BILLING_TIMEZONE_STRING);

    private static final int BILLING_STATUS_OK_CODE = 0;
    private static final String BILLING_STATUS_OK_MESSAGE = "OK";
    private static final DateTimeFormatter BILLING_DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss");

    private final TvmWrapper tvm;
    private final boolean tvmEnabled;
    private final String tvmDestinationAlias;

    private final XmlRpcRequestXmlConverter requestXmlConverter;
    private final XmlRpcResponseXmlConverter responseXmlConverter;
    private final ObjectMapper pojoBinder;

    public BillingApiClient(AsyncHttpClientWrapper asyncHttpClient, BillingApiProperties config, TvmWrapper tvm) {
        super(asyncHttpClient, config, null);
        this.tvm = tvm;
        this.tvmEnabled = config.getTvmEnabled() == Boolean.TRUE;
        this.tvmDestinationAlias = config.getTvmDestinationAlias();
        Preconditions.checkArgument(!tvmEnabled || tvm != null,
                "Enabled destination tvm support requires a not null TvmWrapper");

        XmlRpcConverters converters = XmlRpcHelper.createXmlConverters();
        this.requestXmlConverter = converters.getRequestConverter();
        this.responseXmlConverter = converters.getResponseConverter();
        pojoBinder = new ObjectMapper()
                .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
                .configure(DeserializationFeature.READ_DATE_TIMESTAMPS_AS_NANOSECONDS, false)
                .registerModule(new JavaTimeModule());
    }

    public void updatePayment(long serviceId, long transactionId, Instant payoutReadyDt) {
        log.debug("Calling updatePayment for service {}, transaction {}, payoutReadyAt {}",
                serviceId, transactionId, payoutReadyDt);
        LocalDateTime payoutAtBillingTz = payoutReadyDt.atZone(BILLING_TIMEZONE).toLocalDateTime();
        XmlRpcRequest req = new XmlRpcRequest("Balance.UpdatePayment", List.of(
                Map.of("ServiceID", serviceId, "TransactionID", transactionId),
                Map.of("PayoutReadyDT", BILLING_DATE_FORMAT.format(payoutAtBillingTz))
        ));
        BillingStatusResponse rsp = sync(sendRequest("POST", "", req, BillingStatusResponse.class, "updatePayment"));
        if (rsp.getCode() != BILLING_STATUS_OK_CODE) {
            log.error("updatePayment call failed: {}", rsp);
            throw new RuntimeException("updatePayment call failed: code=" + rsp.getCode() + ", message=" +
                    rsp.getMessage() + ", transactionId=" + transactionId + ", payoutReadyDt=" + payoutReadyDt);
        }
        if (!BILLING_STATUS_OK_MESSAGE.equals(rsp.getMessage())) {
            log.warn("Unexpected OK response message: {}", rsp.getMessage());
        } else {
            log.debug("The transaction has been successfully updated");
        }
    }

    public List<BillingClientContract> getClientContracts(long clientId) {
        log.debug("Calling getClientContracts for clientId {}", clientId);
        BillingClientContractsRequest params = BillingClientContractsRequest.builder()
                .clientId(clientId)
                .build();
        XmlRpcRequest req = new XmlRpcRequest("Balance.GetClientContracts", List.of(params));
        return sync(sendRequest("POST", "", req, Responses.ContractsList.class, "getClientContracts"));
    }

    public List<BillingPerson> getClientPersons(long clientId) {
        log.debug("Calling getClientPersons for clientId {}", clientId);
        XmlRpcRequest req = new XmlRpcRequest("Balance.GetClientPersons", List.of(clientId));
        return sync(sendRequest("POST", "", req, Responses.PersonsList.class, "getClientPersons"));
    }

    public BillingPerson getPerson(long personId) {
        log.debug("Calling getPerson for id {}", personId);
        BillingPersonRequest params = BillingPersonRequest.builder()
                .id(personId)
                .build();
        XmlRpcRequest req = new XmlRpcRequest("Balance.GetPerson", List.of(params));
        Responses.PersonsList personsList = sync(sendRequest("POST", "", req, Responses.PersonsList.class, "getPerson"
        ));
        if (personsList.isEmpty()) {
            return null;
        }
        Preconditions.checkState(personsList.size() == 1, "Expected to find a single person");
        return personsList.get(0);
    }

    public BillingClient getClient(long clientId) {
        List<BillingClient> clientList = getClients(List.of(clientId));
        if (clientList.isEmpty()) {
            return null;
        }
        Preconditions.checkState(clientList.size() == 1, "Expected to find a single client");
        return clientList.get(0);
    }

    public List<BillingClient> getClients(List<Long> clientIds) {
        log.debug("Calling getClientByIdBatch for clientIds {}", clientIds);
        XmlRpcRequest req = new XmlRpcRequest("Balance.GetClientByIdBatch", List.of(clientIds));
        return sync(sendRequest("POST", "", req, Responses.ClientList.class, "getClientByIdBatch"));
    }

    public long createClient(long operatorId, BillingClient client) {
        log.info("Creating a new billing client for '{}'", client.getName());
        Preconditions.checkArgument(client.getClientId() == null,
                "Only new clients are supported in this method; null clientId is expected");
        BillingCreateClientResponse response = syncXmlRpcRequest("Balance2.CreateClient",
                List.of(operatorId, client), BillingCreateClientResponse.class);
        Preconditions.checkState(response.getStatus() == BILLING_STATUS_OK_CODE,
                "Unexpected CreateClient response; status %s, message [%s]",
                response.getStatus(), response.getMessage());
        log.info("A new billing client has been created: id {}", response.getClientId());
        return response.getClientId();
    }

    public long createPerson(long operatorId, BillingUrPerson person) {
        log.debug("Creating a new billing person for clientId {}; name '{}'",
                person.getClientId(), person.getName());
        Preconditions.checkArgument(person.getPersonId() == null,
                "Only new persons are supported in this method; null personId is expected");
        Number personId = syncXmlRpcRequest("Balance2.CreatePerson",
                List.of(operatorId, person), Number.class);
        log.info("A new billing person has been created: id {}", personId);
        return personId.longValue();
    }

    public BillingCreateContractResponse createContract(long operatorId, BillingContract contract) {
        log.debug("Creating a new billing contract for clientId {} and personId {}",
                contract.getClientId(), contract.getPersonId());
        BillingCreateContractResponse result = syncXmlRpcRequest("Balance2.CreateOffer",
                List.of(operatorId, contract), BillingCreateContractResponse.class);
        log.info("A new billing contract has been created: client id {}, contract id {}, external id {}",
                contract.getClientId(), result.getContractId(), result.getExternalId());
        return result;
    }

    public void updateClient(long operatorId, BillingClient client) {
        log.info("Updating the billing client of '{}'; clientId {}", client.getName(), client.getClientId());
        Preconditions.checkArgument(client.getClientId() != null,
                "the clientId field should be set in order to perform the update");
        BillingCreateClientResponse response = syncXmlRpcRequest("Balance2.CreateClient",
                List.of(operatorId, client), BillingCreateClientResponse.class);
        Preconditions.checkState(response.getStatus() == BILLING_STATUS_OK_CODE,
                "Unexpected CreateClient response; status %s, message [%s]",
                response.getStatus(), response.getMessage());
        log.info("The billing client has been updated: clientId {}", response.getClientId());
        Preconditions.checkState(client.getClientId().equals(response.getClientId()),
                "returned clientId mismatch: expected %s -vs- actual %s",
                client.getClientId(), response.getClientId());
    }

    public void updatePerson(long operatorId, BillingUrPerson person) {
        log.debug("Updating the billing person of '{}'; clientId {}, personId {}",
                person.getName(), person.getClientId(), person.getPersonId());
        Preconditions.checkArgument(person.getPersonId() != null,
                "the personId field should be set in order to perform the update");
        Number personId = syncXmlRpcRequest("Balance2.CreatePerson",
                List.of(operatorId, person), Number.class);
        log.info("The billing person has been updated: clientId {}, personId {}", person.getClientId(), personId);
        Preconditions.checkState(person.getPersonId().equals(personId.longValue()),
                "returned personId mismatch: expected %s -vs- actual %s",
                person.getPersonId(), personId);
    }

    public BillingCheckRUBankAccountResponse checkRUBankAccount(String bik, String account) {
        log.debug("Calling checkRUBankAccount for bik {} and account {}", bik, account);
        return syncXmlRpcRequest("Balance.CheckRUBankAccount", List.of(bik, account), BillingCheckRUBankAccountResponse.class);
    }

    // No updateContract method here.
    // The Balance.UpdateContract API hasn't been released to Prod yet.

    private <T> T syncXmlRpcRequest(String method, List<Object> params, Class<T> responseType) {
        log.debug("Calling XML RPC: method {}, params: {}", method, params);
        XmlRpcRequest request = new XmlRpcRequest(method, params);
        return sync(sendRequest("POST", "", request, responseType, method));
    }

    @SuppressWarnings("unchecked")
    protected <T> String serializeRequest(T request) {
        if (request == null) {
            return null;
        }
        Preconditions.checkArgument(request instanceof XmlRpcRequest,
                "Illegal request type: %s", request.getClass().getName());
        XmlRpcRequest xmlRpcRequest = (XmlRpcRequest) request;

        List<Object> basicTypeParams = (List<Object>)
                pojoBinder.convertValue(xmlRpcRequest.getParameters(), Object.class);
        basicTypeParams = (List<Object>) XmlRpcHelper.convertToBasicType(basicTypeParams);
        return requestXmlConverter.convertToXml(xmlRpcRequest.getMethodName(), basicTypeParams);
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        if (tvmEnabled) {
            rb.addHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(),
                    tvm.getServiceTicket(tvmDestinationAlias));
        }
        return rb;
    }

    protected <T> T parseSuccessfulContent(Response response, Class<T> contentType) {
        try {
            String body = response.getResponseBody();
            if (Strings.isNullOrEmpty(body) || contentType == null) {
                return null;
            }
            Object data = responseXmlConverter.convertFromXml(body);
            traverseAndFixAllDates(data);
            return pojoBinder.convertValue(data, contentType);
        } catch (XmlRpcRuntimeException e) {
            log.warn("XMC RPC error has occurred", e);
            throw e;
        } catch (Exception e) {
            log.error("Unable to parse response", e);
            throw new HttpApiParsingException("Unable to parse response", e, response.getStatusCode(),
                    response.getResponseBody());
        }
    }

    @SuppressWarnings("unchecked")
    private void traverseAndFixAllDates(Object xmlRpcValue) {
        Class<?> type = xmlRpcValue.getClass();
        if (type.isArray() && !type.getComponentType().isPrimitive()) {
            int length = Array.getLength(xmlRpcValue);
            for (int i = 0; i < length; i++) {
                // Changing array type is unsupported at the moment (Date[] -> LocalDateTime[])
                traverseAndFixAllDates(Array.get(xmlRpcValue, i));
            }
        } else if (xmlRpcValue instanceof Map) {
            for (Map.Entry<?, Object> entry : ((Map<?, Object>) xmlRpcValue).entrySet()) {
                if (entry.getValue() instanceof Date) {
                    entry.setValue(convertDate((Date) entry.getValue()));
                } else {
                    traverseAndFixAllDates(entry.getValue());
                }
            }
        }
        // else: some other value, doesn't matter as we traverse only structs and lists and look for dates
    }

    private LocalDateTime convertDate(Date date) {
        SimpleDateFormat fmt = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss");
        return LocalDateTime.parse(fmt.format(date));
    }

    private static class Responses {
        public static class ContractsList extends ArrayList<BillingClientContract> {
        }

        public static class PersonsList extends ArrayList<BillingPerson> {
        }

        public static class ClientList extends ArrayList<BillingClient> {
        }
    }
}
