package ru.yandex.travel.integration.balance;

import java.net.URL;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.utils.URIBuilder;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.AsyncHttpClientBase;
import ru.yandex.travel.commons.http.apiclient.HttpMethod;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.integration.balance.csv.BankOrderDetailParser;
import ru.yandex.travel.integration.balance.csv.BankOrderParser;
import ru.yandex.travel.integration.balance.model.csv.BankOrder;
import ru.yandex.travel.integration.balance.model.csv.BankOrderDetail;
import ru.yandex.travel.tvm.TvmWrapper;

/**
 * https://wiki.yandex-team.ru/balance/httpapi/newpaymentstat/
 */
@Slf4j
public class BillingCsvApiClient extends AsyncHttpClientBase {
    public static final ZoneId BILLING_TIME_ZONE_ID = ZoneId.of("Europe/Moscow");

    private static final DateTimeFormatter BALANCE_TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");

    private static final String METHOD_GET_PAYMENT_HEADERS = "get_payment_headers";
    private static final String METHOD_GET_PAYMENT_BATCH_DETAILS = "get_payment_batch_details";

    private final String balanceUrl;

    private final TvmWrapper tvm;
    private final boolean tvmEnabled;
    private final String tvmDestinationAlias;

    public BillingCsvApiClient(
            AsyncHttpClientWrapper asyncHttpClient,
            BillingApiProperties config,
            TvmWrapper tvm
    ) {
        super(asyncHttpClient, config);
        this.balanceUrl = config.getBaseUrl();

        this.tvm = tvm;
        this.tvmEnabled = config.getTvmEnabled() == Boolean.TRUE;
        this.tvmDestinationAlias = config.getTvmDestinationAlias();
        Preconditions.checkArgument(!tvmEnabled || tvm != null,
                "Enabled destination tvm support requires a not null TvmWrapper");
    }

    public List<BankOrder> getBankOrders(Long serviceId, LocalDate fromDateInclusive, LocalDate toDateExclusive) {
        final Map<String, String> params = new HashMap<>();
        params.put("service_id", String.valueOf(serviceId));
        params.put("from_trantime", BALANCE_TIMESTAMP_FORMAT.format(fromDateInclusive.atStartOfDay()));
        params.put("to_trantime", BALANCE_TIMESTAMP_FORMAT.format(toDateExclusive.atStartOfDay()));

        final String path = generateBalanceUrl(METHOD_GET_PAYMENT_HEADERS, params);

        final BankOrderParser bankOrderParser = new BankOrderParser(serviceId);
        return sync(sendRequest(
                HttpMethod.GET,
                path,
                null,
                ignore -> null,
                (Response r) -> bankOrderParser.parse(r.getResponseBodyAsStream()),
                METHOD_GET_PAYMENT_HEADERS
        ));
    }

    public List<BankOrderDetail> getBankOrderDetails(String paymentBatchId) {
        final Map<String, String> params = new HashMap<>();
        params.put("payment_batch_id", paymentBatchId);

        final String path = generateBalanceUrl(METHOD_GET_PAYMENT_BATCH_DETAILS, params);

        final BankOrderDetailParser bankOrderDetailParser = new BankOrderDetailParser(paymentBatchId);
        return sync(sendRequest(
                HttpMethod.GET,
                path,
                null,
                ignore -> null,
                (Response r) -> bankOrderDetailParser.parse(r.getResponseBodyAsStream()),
                METHOD_GET_PAYMENT_BATCH_DETAILS
        ));
    }

    private String generateBalanceUrl(final String method, final Map<String, String> params) {
        try {
            final URIBuilder uriBuilder = new URIBuilder(balanceUrl + "/httpapitvm/" + method);
            params.forEach(uriBuilder::addParameter);
            final URL url = new URL(uriBuilder.build().toString());
            return url.getPath() + "?" + url.getQuery();
        } catch (final Exception ex) {
            throw new RuntimeException(String.format("Invalid Balance url (params: %s)", params), ex);
        }
    }

    @Override
    protected RequestBuilder createBaseRequestBuilder(HttpMethod method, String path, String body) {
        RequestBuilder rb = super.createBaseRequestBuilder(method, path, body);
        if (tvmEnabled) {
            rb.addHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(),
                    tvm.getServiceTicket(tvmDestinationAlias));
        }
        return rb;
    }
}
