package ru.yandex.travel.integration.balance.csv;

import java.time.LocalDateTime;

import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVRecord;
import org.apache.commons.csv.QuoteMode;

import ru.yandex.travel.integration.balance.model.csv.BankOrderDetail;
import ru.yandex.travel.integration.balance.model.csv.BillingCurrency;
import ru.yandex.travel.integration.balance.model.csv.PaymentType;
import ru.yandex.travel.integration.balance.model.csv.TransactionType;


public class BankOrderDetailParser extends BaseResultCsvParser<BankOrderDetail> {

    private static final String TRUST_PAYMENT_ID = "TRUST_PAYMENT_ID";
    private static final String TRUST_REFUND_ID = "TRUST_REFUND_ID";
    private static final String SERVICE_ORDER_ID = "SERVICE_ORDER_ID";
    private static final String SUM = "SUM";
    private static final String CURRENCY = "CURRENCY";
    private static final String PAYMENT_TYPE = "PAYMENT_TYPE";
    private static final String TRANSACTION_TYPE = "TRANSACTION_TYPE";
    private static final String HANDLING_TIME = "HANDLINGTIME";
    private static final String PAYMENT_TIME = "PAYMENTTIME";
    private static final String CONTRACT_ID = "CONTRACT_ID";
    private static final String AGENCY_COMMISSION = "YANDEX_REWARD";

    private final String paymentBatchId;

    public BankOrderDetailParser(String paymentBatchId) {
        super(CSVFormat.DEFAULT
                .withDelimiter('\t')
                .withQuoteMode(QuoteMode.MINIMAL)
                .withIgnoreEmptyLines()
                .withNullString("")
                .withFirstRecordAsHeader()
        );
        this.paymentBatchId = paymentBatchId;
    }

    @Override
    protected BankOrderDetail convert(CSVRecord record) {
        return BankOrderDetail.builder()
                .paymentBatchId(paymentBatchId)
                .trustPaymentId(record.get(TRUST_PAYMENT_ID))
                .trustRefundId(record.get(TRUST_REFUND_ID))
                .serviceOrderId(record.get(SERVICE_ORDER_ID))
                .sum(CsvValueParser.parseBigDecimal(record.get(SUM)).orElseThrow())
                .currency(BillingCurrency.valueOf(record.get(CURRENCY)))
                .paymentType(PaymentType.forValue(record.get(PAYMENT_TYPE)))
                .transactionType(TransactionType.forValue(record.get(TRANSACTION_TYPE)))
                .handlingTime(CsvValueParser.parseLocalDateTime(record.get(HANDLING_TIME)).map(LocalDateTime::toLocalDate).orElseThrow())
                .paymentTime(CsvValueParser.parseLocalDateTime(record.get(PAYMENT_TIME)).map(LocalDateTime::toLocalDate).orElseThrow())
                .contractId(CsvValueParser.parseLong(record.get(CONTRACT_ID)).orElseThrow())
                .agencyCommission(CsvValueParser.parseBigDecimal(record.get(AGENCY_COMMISSION)).orElseThrow())
                .build();
    }
}
