package ru.yandex.travel.integration.balance.csv;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.Optional;
import java.util.TimeZone;

import org.apache.commons.lang3.StringUtils;


public final class CsvValueParser {

    private static final String NULL_VALUE = "null";
    private static final DateTimeFormatter BALANCE_TIMESTAMP_FORMAT = DateTimeFormatter.ofPattern("yyyyMMddHHmmss")
            .withZone(TimeZone.getDefault().toZoneId());

    public static Optional<String> getNullableString(String value) {
        if (StringUtils.isBlank(value) || NULL_VALUE.equalsIgnoreCase(value)) {
            return Optional.empty();
        } else {
            return Optional.of(value);
        }
    }

    static Optional<Long> parseLong(String value) {
        return getNullableString(value).map(Long::parseLong);
    }

    static Optional<BigDecimal> parseBigDecimal(String value) {
        return getNullableString(value).map(BigDecimal::new);
    }

    static Optional<Integer> parseInteger(String value) {
        return getNullableString(value).map(Integer::parseInt);
    }

    static Optional<Instant> parseInstant(String value) {
        return getNullableString(value).map(s -> LocalDateTime.parse(s, BALANCE_TIMESTAMP_FORMAT)
                .atZone(ZoneId.systemDefault())
                .toInstant()
        );
    }

    static Optional<LocalDateTime> parseLocalDateTime(String value) {
        return getNullableString(value).map(s -> LocalDateTime.parse(s, BALANCE_TIMESTAMP_FORMAT));
    }
}
