package ru.yandex.travel.commons.concurrent;

import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;

public abstract class AbstractSemaphoreWithFlags {

    private final AtomicLong value = new AtomicLong();

    protected final Logger logger;

    protected final String name;

    public AbstractSemaphoreWithFlags(Logger logger, String name) {
        this.logger = logger;
        this.name = name;
    }

    // Try to start a new task. Returns true if the task is started (i.e. by the time of the call we were not shutting down)
    public boolean acquire() {
        while (true) {
            long oldValue = value.get();

            int permits = permitsFromValue(oldValue);
            int flags = flagsFromValue(oldValue);

            if (!allowPermitAcquiring(permits, flags)) {
                return false;
            }

            ++permits;
            long newValue = valueFrom(permits, flags);
            if (value.compareAndSet(oldValue, newValue)) {
                if (logger != null) {
                    logger.debug("{} semaphore: tasks running: {}", name, permits);
                }
                return true;
            }
        }
    }

    public void release() {
        while (true) {
            long oldValue = value.get();
            int permits = permitsFromValue(oldValue);
            int flags = flagsFromValue(oldValue);
            --permits;
            Preconditions.checkState(permits >= 0, "Semaphore is negative");
            long newValue = valueFrom(permits, flags);
            if (value.compareAndSet(oldValue, newValue)) {
                if (logger != null) {
                    logger.debug("{} semaphore: tasks running: {}", name, permits);
                }
                onPermitReleased(permits, flags);
                return;
            }
        }
    }

    protected final void updateFlags(Function<Integer, Integer> changeFlagsFunction, Consumer<Integer> onMaskChanged) {
        while (true) {
            long oldValue = value.get();
            int permits = permitsFromValue(oldValue);
            int flags = flagsFromValue(oldValue);
            int newFlags = changeFlagsFunction.apply(flags);
            long newValue = valueFrom(permits, newFlags);
            if (value.compareAndSet(oldValue, newValue)) {
                onMaskChanged.accept(permits);
                return;
            }
        }
    }

    protected final int getCurrentFlags() {
        long oldValue = value.get();
        return flagsFromValue(oldValue);
    }

    protected final int getCurrentPermits() {
        long oldValue = value.get();
        return permitsFromValue(oldValue);
    }

    protected <T> T produceFromPermitsAndFlags(BiFunction<Integer, Integer, T> f) {
        long oldValue = value.get();
        return f.apply(permitsFromValue(oldValue), flagsFromValue(oldValue));
    }

    protected final boolean isFlagPresent(int flags, int flagMask) {
        return (flags & flagMask) == flagMask;
    }


    private int permitsFromValue(long value) {
        return (int) (value >> 32);
    }

    private int flagsFromValue(long value) {
        return (int) (value);
    }

    private long valueFrom(int permits, int flags) {
        return (long) permits << 32 | flags & 0xFFFFFFFFL;
    }

    protected abstract boolean allowPermitAcquiring(int count, int flags);

    protected abstract void onPermitReleased(int count, int flags);

}
