package ru.yandex.travel.commons.concurrent;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import org.slf4j.MDC;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.retry.Retry;

public class FutureUtils {
    public static <T> CompletableFuture<T> buildCompletableFuture(ListenableFuture<T> source) {
        CompletableFuture<T> future = new CompletableFuture<>() {
            @Override
            public boolean cancel(boolean mayInterruptIfRunning) {
                boolean cancelResult = source.cancel(mayInterruptIfRunning);
                super.cancel(mayInterruptIfRunning);
                return cancelResult;
            }
        };

        Futures.addCallback(source, new FutureCallback<>() {
            @Override
            public void onSuccess(@Nullable T result) {
                future.complete(result);
            }

            @Override
            public void onFailure(Throwable t) {
                future.completeExceptionally(t);
            }
        }, MoreExecutors.directExecutor() );
        return future;
    }

    public static <T> CompletableFuture<T> buildExceptional(Throwable ex) {
        CompletableFuture<T> future = new CompletableFuture<>();
        future.completeExceptionally(ex);
        return future;
    }

    public static <T, E extends Exception> CompletableFuture<T> handleExceptionOfType(CompletableFuture<T> future,
                                                                                      Class<E> exceptionClass,
                                                                                      Function<E, T> handler) {
        return future.exceptionally(t -> {
            if (t instanceof Exception) {
                Exception cause = Retry.unwrapExecutionException((Exception) t);
                if (exceptionClass.isInstance(cause)) {
                    E castedCause = exceptionClass.cast(cause);
                    return handler.apply(castedCause);
                } else {
                    if (t instanceof RuntimeException) {
                        throw (RuntimeException) t;
                    } else {
                        throw new RuntimeException(t);
                    }
                }
            } else {
                throw new RuntimeException(t);
            }
        });
    }

    public static <T,R> Function<T, R> withMdc(Function<T, R> function) {
        Map<String, String> mdc = MDC.getCopyOfContextMap();
        return (t) -> {
            try (var ignoredMdc = NestedMdc.nestedMdc(mdc)) {
                return function.apply(t);
            }
        };
    }

    /**
     * Ensures non-blocking access.
     */
    public static <T> T getCompleted(CompletableFuture<T> future) throws ExecutionException, InterruptedException {
        Preconditions.checkArgument(future.isDone(), "The future has to be completed before passing to this method");
        return future.get();
    }

    /**
     * Ensures non-blocking access.
     */
    public static <T> T joinCompleted(CompletableFuture<T> future) {
        Preconditions.checkArgument(future.isDone(), "The future has to be completed before passing to this method");
        return future.join();
    }
}
