package ru.yandex.travel.commons.concurrent;

import org.slf4j.Logger;

public class TerminationSemaphore extends AbstractSemaphoreWithFlags {
    public static final int SHUTTING_DOWN_MASK = 1;

    public TerminationSemaphore() {
        this(null, null);
    }

    public TerminationSemaphore(Logger logger, String name) {
        super(logger, name);
    }

    private final Object syncRoot = new Object();

    @Override
    protected boolean allowPermitAcquiring(int count, int flags) {
        boolean isShuttingDown = (flags & SHUTTING_DOWN_MASK) == SHUTTING_DOWN_MASK;
        return !isShuttingDown;
    }

    @Override
    protected void onPermitReleased(int count, int flags) {
        boolean isShuttingDown = (flags & SHUTTING_DOWN_MASK) == SHUTTING_DOWN_MASK;
        if (isShuttingDown && (count == 0)) {
            synchronized (syncRoot) {
                syncRoot.notifyAll();
            }
        }
    }

    // Schedule termination.
    // Termination actually happens if the processing has stopped, i.e. the number of running tasks was 0.
    public void shutdown() {
        updateFlags(
                (currenMask) -> currenMask | SHUTTING_DOWN_MASK,
                (permits) -> {
                    if (permits == 0) {
                        synchronized (syncRoot) {
                            syncRoot.notifyAll();
                        }
                    }
                }
        );
    }

    public boolean hasTerminated() {
        return produceFromPermitsAndFlags(
                (permits, flags) -> ((flags & SHUTTING_DOWN_MASK) == SHUTTING_DOWN_MASK) && (permits == 0)
        );
    }

    public boolean isShutDown() {
        return (getCurrentFlags() & SHUTTING_DOWN_MASK) == SHUTTING_DOWN_MASK;
    }

    public void awaitTermination() throws InterruptedException {
        int permits = getCurrentPermits();
        if (logger != null) {
            logger.info("{} termination semaphore: Waiting for {} tasks to complete", name, permits);
        }
        if (!hasTerminated()) {
            synchronized (syncRoot) {
                if (!hasTerminated()) {
                    syncRoot.wait();
                }
            }
        }
    }
}
