package ru.yandex.travel.commons.enums;

import java.util.Optional;
import java.util.function.Function;

import com.google.common.collect.ImmutableBiMap;
import lombok.AllArgsConstructor;

@AllArgsConstructor
public class EnumIndex<K, V extends Enum<V>> {
    private final Class<V> enumClass;
    private final ImmutableBiMap<K, V> valueToEnumIndex;

    public static <K, V extends Enum<V> & ValueBasedEnum<K>> EnumIndex<K, V> create(Class<V> enumClass) {
        return create(enumClass, ValueBasedEnum::getValue);
    }

    public static <V extends Enum<V> & IntBasedEnum> EnumIndex<Integer, V> createIntBased(Class<V> enumClass) {
        return create(enumClass, IntBasedEnum::getValue);
    }

    public static <K, V extends Enum<V>> EnumIndex<K, V> create(Class<V> enumClass, Function<V, K> keyMapper) {
        ImmutableBiMap.Builder<K, V> mapBuilder = ImmutableBiMap.builder();
        for (V value : enumClass.getEnumConstants()) {
            mapBuilder.put(keyMapper.apply(value), value);
        }
        return new EnumIndex<>(enumClass, mapBuilder.build());
    }

    /**
     * To be used with getByValueOrNull
     *
     * @param enumClass
     * @param keyMapper returns Optional<Key>. If it's empty than the value is not mapped.
     * @param <K>       key type
     * @param <V>       value type
     * @return Key-value index, but not all original enum values may be present as some values might generate empty keys
     */
    public static <K, V extends Enum<V>> EnumIndex<K, V> createSelective(Class<V> enumClass,
                                                                         Function<V, Optional<K>> keyMapper) {
        ImmutableBiMap.Builder<K, V> mapBuilder = ImmutableBiMap.builder();
        for (V value : enumClass.getEnumConstants()) {
            Optional<K> maybeKey = keyMapper.apply(value);
            if (maybeKey.isEmpty()) {
                continue;
            }
            mapBuilder.put(maybeKey.get(), value);
        }
        return new EnumIndex<>(enumClass, mapBuilder.build());
    }

    public V getByValue(K value) {
        V enumObject = valueToEnumIndex.get(value);
        if (enumObject == null) {
            throw new IllegalArgumentException(String.format(
                    "No such enum value of class %s: %s",
                    enumClass.getName(), value
            ));
        }
        return enumObject;
    }

    public V getByValueOrNull(K value) {
        return valueToEnumIndex.get(value);
    }

    public V getByValueOrDefault(K value, V defaultValue) {
        return valueToEnumIndex.getOrDefault(value, defaultValue);
    }
}
